import jsPDF from "jspdf";
import autoTable from "jspdf-autotable";

interface P9MonthData {
  month: string; // e.g. "January"
  basicSalary: number;
  benefits: number; // housing + transport + other allowances + overtime + bonus
  grossPay: number;
  thirtyPercentOfGross: number; // 30% of gross (for housing benefit limit)
  actualContribution: number; // NSSF employee contribution
  permissibleLimit: number; // NSSF statutory limit (currently 1080 per month for Tier I)
  definedContribution: number; // lesser of actual and permissible
  housingLevy: number; // housing levy employee
  taxablePay: number; // gross - defined contribution - housing levy
  taxCharged: number; // tax on taxable pay
  personalRelief: number; // 2400 per month
  insuranceRelief: number; // NHIF/SHA relief
  payeDeducted: number; // actual PAYE deducted
}

interface P9Data {
  year: string;
  employerName: string;
  employerPin: string;
  employeeName: string;
  employeePin: string;
  employeeNumber: string;
  months: P9MonthData[];
}

const MONTH_NAMES = [
  "January", "February", "March", "April", "May", "June",
  "July", "August", "September", "October", "November", "December",
];

const fmt = (n: number) => n === 0 ? "—" : Number(n).toLocaleString("en-KE", { minimumFractionDigits: 2, maximumFractionDigits: 2 });

export function generateP9PDF(data: P9Data) {
  const doc = new jsPDF({ orientation: "landscape", unit: "mm", format: "a4" });

  // Header
  doc.setFontSize(14);
  doc.setFont("helvetica", "bold");
  doc.text("TAX DEDUCTION CARD (P9)", 148.5, 14, { align: "center" });
  doc.setFontSize(9);
  doc.setFont("helvetica", "normal");
  doc.text("REPUBLIC OF KENYA — KENYA REVENUE AUTHORITY", 148.5, 20, { align: "center" });

  doc.setDrawColor(0, 80, 120);
  doc.setLineWidth(0.5);
  doc.line(10, 23, 287, 23);

  // Employer and Employee Info
  const infoData = [
    ["Employer's Name:", data.employerName, "Employee's Name:", data.employeeName],
    ["Employer's PIN:", data.employerPin, "Employee's PIN:", data.employeePin],
    ["Year of Income:", data.year, "Employee No:", data.employeeNumber],
  ];

  autoTable(doc, {
    startY: 26,
    head: [],
    body: infoData,
    theme: "plain",
    styles: { fontSize: 8, cellPadding: 1.5 },
    columnStyles: {
      0: { fontStyle: "bold", cellWidth: 35 },
      1: { cellWidth: 65 },
      2: { fontStyle: "bold", cellWidth: 35 },
      3: { cellWidth: 65 },
    },
  });

  const afterInfo = (doc as any).lastAutoTable.finalY + 4;

  // Section A header
  doc.setFontSize(9);
  doc.setFont("helvetica", "bold");
  doc.text("SECTION A: EMPLOYMENT INCOME", 10, afterInfo);

  // Build monthly rows
  const monthlyRows: any[][] = [];
  for (let i = 0; i < 12; i++) {
    const m = data.months.find((md) => md.month === MONTH_NAMES[i]);
    if (m) {
      monthlyRows.push([
        MONTH_NAMES[i],
        fmt(m.basicSalary),
        fmt(m.benefits),
        fmt(m.grossPay),
        fmt(m.definedContribution),
        fmt(m.housingLevy),
        fmt(m.taxablePay),
        fmt(m.taxCharged),
        fmt(m.personalRelief),
        fmt(m.insuranceRelief),
        fmt(m.payeDeducted),
      ]);
    } else {
      monthlyRows.push([
        MONTH_NAMES[i], "—", "—", "—", "—", "—", "—", "—", "—", "—", "—",
      ]);
    }
  }

  // Totals
  const totals = data.months.reduce(
    (acc, m) => ({
      basicSalary: acc.basicSalary + m.basicSalary,
      benefits: acc.benefits + m.benefits,
      grossPay: acc.grossPay + m.grossPay,
      definedContribution: acc.definedContribution + m.definedContribution,
      housingLevy: acc.housingLevy + m.housingLevy,
      taxablePay: acc.taxablePay + m.taxablePay,
      taxCharged: acc.taxCharged + m.taxCharged,
      personalRelief: acc.personalRelief + m.personalRelief,
      insuranceRelief: acc.insuranceRelief + m.insuranceRelief,
      payeDeducted: acc.payeDeducted + m.payeDeducted,
    }),
    {
      basicSalary: 0, benefits: 0, grossPay: 0, definedContribution: 0,
      housingLevy: 0, taxablePay: 0, taxCharged: 0, personalRelief: 0,
      insuranceRelief: 0, payeDeducted: 0,
    }
  );

  monthlyRows.push([
    { content: "TOTAL", styles: { fontStyle: "bold" } },
    { content: fmt(totals.basicSalary), styles: { fontStyle: "bold" } },
    { content: fmt(totals.benefits), styles: { fontStyle: "bold" } },
    { content: fmt(totals.grossPay), styles: { fontStyle: "bold" } },
    { content: fmt(totals.definedContribution), styles: { fontStyle: "bold" } },
    { content: fmt(totals.housingLevy), styles: { fontStyle: "bold" } },
    { content: fmt(totals.taxablePay), styles: { fontStyle: "bold" } },
    { content: fmt(totals.taxCharged), styles: { fontStyle: "bold" } },
    { content: fmt(totals.personalRelief), styles: { fontStyle: "bold" } },
    { content: fmt(totals.insuranceRelief), styles: { fontStyle: "bold" } },
    { content: fmt(totals.payeDeducted), styles: { fontStyle: "bold" } },
  ]);

  autoTable(doc, {
    startY: afterInfo + 3,
    head: [[
      "Month",
      "Basic\nSalary\n(a)",
      "Benefits/\nAllowances\n(b)",
      "Gross Pay\n(c=a+b)",
      "Defined\nContrib.\n(d)",
      "Housing\nLevy\n(e)",
      "Taxable\nPay\n(f=c-d-e)",
      "Tax\nCharged\n(g)",
      "Personal\nRelief\n(h)",
      "Insurance\nRelief\n(i)",
      "PAYE\n(j=g-h-i)",
    ]],
    body: monthlyRows,
    theme: "grid",
    headStyles: {
      fillColor: [20, 60, 90],
      fontSize: 7,
      halign: "center",
      valign: "middle",
      cellPadding: 1,
    },
    styles: { fontSize: 7.5, cellPadding: 1.5 },
    columnStyles: {
      0: { cellWidth: 22 },
      1: { halign: "right", cellWidth: 24 },
      2: { halign: "right", cellWidth: 24 },
      3: { halign: "right", cellWidth: 24 },
      4: { halign: "right", cellWidth: 24 },
      5: { halign: "right", cellWidth: 22 },
      6: { halign: "right", cellWidth: 26 },
      7: { halign: "right", cellWidth: 24 },
      8: { halign: "right", cellWidth: 22 },
      9: { halign: "right", cellWidth: 22 },
      10: { halign: "right", cellWidth: 24 },
    },
  });

  const afterTable = (doc as any).lastAutoTable.finalY + 8;

  // Footer
  doc.setFontSize(7.5);
  doc.setFont("helvetica", "normal");
  doc.setTextColor(100);
  doc.text("NOTE: This is a computer-generated P9 Tax Deduction Card. It does not require a signature.", 10, afterTable);
  doc.text(`Generated on ${new Date().toLocaleDateString("en-KE", { year: "numeric", month: "long", day: "numeric" })}`, 10, afterTable + 4);
  doc.text("Personal Relief: KES 2,400/month (KES 28,800/year) | Insurance Relief: 15% of NHIF/SHA contribution (max KES 5,000/month)", 10, afterTable + 8);

  doc.save(`P9-${data.employeeName.replace(/\s+/g, "_")}-${data.year}.pdf`);
}

export function buildP9MonthData(payrollRecords: any[]): P9MonthData[] {
  return payrollRecords.map((r) => {
    const basicSalary = Number(r.basic_salary);
    const benefits = Number(r.housing_allowance) + Number(r.transport_allowance) +
      Number(r.other_allowances) + Number(r.overtime_pay) + Number(r.bonus);
    const grossPay = Number(r.gross_pay);
    const nssfEmployee = Number(r.nssf_employee);
    const housingLevy = Number(r.housing_levy_employee || 0);
    const paye = Number(r.paye);
    const nhif = Number(r.nhif);

    // Defined contribution is NSSF employee contribution
    const definedContribution = nssfEmployee;

    // Taxable pay = gross - NSSF - housing levy
    const taxablePay = grossPay - definedContribution - housingLevy;

    // Tax charged (before reliefs) = PAYE + personal relief + insurance relief
    const personalRelief = 2400;
    const insuranceRelief = Math.min(nhif * 0.15, 5000);
    const taxCharged = paye + personalRelief + insuranceRelief;

    // Determine month name from pay_period (format: "YYYY-MM")
    const [, monthStr] = (r.pay_period as string).split("-");
    const monthIndex = parseInt(monthStr, 10) - 1;
    const month = MONTH_NAMES[monthIndex] || "Unknown";

    return {
      month,
      basicSalary,
      benefits,
      grossPay,
      thirtyPercentOfGross: grossPay * 0.3,
      actualContribution: nssfEmployee,
      permissibleLimit: 1080,
      definedContribution,
      housingLevy,
      taxablePay,
      taxCharged,
      personalRelief,
      insuranceRelief,
      payeDeducted: paye,
    };
  });
}
