// Kenyan payroll statutory calculations (2024/2025)
// SHA replaces NHIF | Housing Levy introduced | NSSF updated | PAYE unchanged

/** PAYE: progressive tax with KES 2,400 personal relief */
export function calculatePAYE(taxableIncome: number): number {
  const personalRelief = 2400;
  let tax = 0;

  if (taxableIncome <= 24000) {
    tax = taxableIncome * 0.10;
  } else if (taxableIncome <= 32333) {
    tax = 24000 * 0.10 + (taxableIncome - 24000) * 0.25;
  } else if (taxableIncome <= 500000) {
    tax = 24000 * 0.10 + 8333 * 0.25 + (taxableIncome - 32333) * 0.30;
  } else if (taxableIncome <= 800000) {
    tax = 24000 * 0.10 + 8333 * 0.25 + 467667 * 0.30 + (taxableIncome - 500000) * 0.325;
  } else {
    tax = 24000 * 0.10 + 8333 * 0.25 + 467667 * 0.30 + 300000 * 0.325 + (taxableIncome - 800000) * 0.35;
  }

  return Math.max(0, tax - personalRelief);
}

/**
 * SHA (Social Health Authority) — replaced NHIF effective Oct 2024.
 * Rate: 2.75% of gross pay (no upper cap).
 */
export function calculateSHA(grossPay: number): number {
  return Math.round(grossPay * 0.0275 * 100) / 100;
}

/** @deprecated Use calculateSHA instead */
export const calculateNHIF = calculateSHA;

/**
 * NSSF (New Rates per NSSF Act 2013, Supreme Court lift 2024):
 * Tier I  — 6% of gross up to KES 7,000  → max KES 420 each
 * Tier II — 6% of gross KES 7,001 – 36,000 → max KES 1,740 each
 */
export function calculateNSSF(grossPay: number): { employee: number; employer: number } {
  const tierICap = 7000;
  const tierIICap = 36000;
  const rate = 0.06;

  const tierI = Math.min(grossPay, tierICap) * rate;
  const tierII = Math.max(0, Math.min(grossPay, tierIICap) - tierICap) * rate;
  const total = Math.round((tierI + tierII) * 100) / 100;

  return { employee: total, employer: total };
}

/**
 * Housing Levy (Affordable Housing Levy Act 2023):
 * 1.5% of gross pay — employee deduction; employer matches 1.5%.
 */
export function calculateHousingLevy(grossPay: number): { employee: number; employer: number } {
  const amount = Math.round(grossPay * 0.015 * 100) / 100;
  return { employee: amount, employer: amount };
}

export interface PayrollInput {
  basicSalary: number;
  housing: number;
  transport: number;
  other: number;
  overtime: number;
  bonus: number;
  loan: number;
  sacco: number;
  otherDed: number;
}

export function computePayroll(input: PayrollInput) {
  const grossPay = input.basicSalary + input.housing + input.transport + input.other + input.overtime + input.bonus;
  const nssf = calculateNSSF(grossPay);
  const housingLevy = calculateHousingLevy(grossPay);

  // Taxable income = gross - NSSF employee - Housing Levy employee (both pre-tax)
  const taxableIncome = grossPay - nssf.employee - housingLevy.employee;
  const paye = calculatePAYE(taxableIncome);
  const sha = calculateSHA(grossPay);

  const totalDeductions =
    paye + sha + nssf.employee + housingLevy.employee + input.loan + input.sacco + input.otherDed;
  const netPay = grossPay - totalDeductions;

  return {
    grossPay,
    paye,
    sha,
    nhif: sha, // backwards-compat alias stored in nhif DB column
    nssf,
    housingLevy,
    taxableIncome,
    totalDeductions,
    netPay,
  };
}
