import jsPDF from "jspdf";
import autoTable from "jspdf-autotable";

interface PayslipPDFData {
  payPeriod: string;
  employeeName: string;
  employeeNumber: string;
  department: string;
  designation: string;
  kraPin: string;
  nhifNumber: string;
  nssfNumber: string;
  bankName: string;
  bankAccount: string;
  basicSalary: number;
  housingAllowance: number;
  transportAllowance: number;
  otherAllowances: number;
  overtimePay: number;
  bonus: number;
  grossPay: number;
  paye: number;
  nhif: number;
  nssfEmployee: number;
  nssfEmployer: number;
  loanDeduction: number;
  saccoDeduction: number;
  otherDeductions: number;
  totalDeductions: number;
  netPay: number;
}

const fmt = (n: number) => `KES ${Number(n).toLocaleString("en-KE", { minimumFractionDigits: 2 })}`;

export function generatePayslipPDF(data: PayslipPDFData) {
  const doc = new jsPDF();

  // Header
  doc.setFontSize(18);
  doc.setFont("helvetica", "bold");
  doc.text("GIDEON WHEELS LTD", 105, 20, { align: "center" });
  doc.setFontSize(11);
  doc.setFont("helvetica", "normal");
  doc.text(`Payslip for ${data.payPeriod}`, 105, 28, { align: "center" });

  doc.setDrawColor(0, 100, 120);
  doc.setLineWidth(0.5);
  doc.line(14, 33, 196, 33);

  // Employee info
  doc.setFontSize(9);
  const info = [
    ["Name", data.employeeName, "Employee #", data.employeeNumber],
    ["Department", data.department, "Designation", data.designation],
    ["KRA PIN", data.kraPin, "NHIF #", data.nhifNumber],
    ["NSSF #", data.nssfNumber, "Bank", `${data.bankName} — ${data.bankAccount}`],
  ];

  autoTable(doc, {
    startY: 37,
    head: [],
    body: info,
    theme: "plain",
    styles: { fontSize: 9, cellPadding: 2 },
    columnStyles: { 0: { fontStyle: "bold", cellWidth: 30 }, 2: { fontStyle: "bold", cellWidth: 30 } },
  });

  const afterInfo = (doc as any).lastAutoTable.finalY + 6;

  // Earnings
  const earnings: (string | string[])[][] = [
    ["Basic Salary", fmt(data.basicSalary)],
  ];
  if (data.housingAllowance > 0) earnings.push(["Housing Allowance", fmt(data.housingAllowance)]);
  if (data.transportAllowance > 0) earnings.push(["Transport Allowance", fmt(data.transportAllowance)]);
  if (data.otherAllowances > 0) earnings.push(["Other Allowances", fmt(data.otherAllowances)]);
  if (data.overtimePay > 0) earnings.push(["Overtime Pay", fmt(data.overtimePay)]);
  if (data.bonus > 0) earnings.push(["Bonus", fmt(data.bonus)]);
  earnings.push([{ content: "Gross Pay", styles: { fontStyle: "bold" } } as any, { content: fmt(data.grossPay), styles: { fontStyle: "bold" } } as any]);

  autoTable(doc, {
    startY: afterInfo,
    head: [["Earnings", "Amount (KES)"]],
    body: earnings,
    theme: "striped",
    headStyles: { fillColor: [20, 60, 90], fontSize: 10 },
    styles: { fontSize: 9 },
    columnStyles: { 1: { halign: "right" } },
  });

  const afterEarnings = (doc as any).lastAutoTable.finalY + 4;

  // Deductions
  const deductions: any[][] = [
    ["PAYE (Income Tax)", fmt(data.paye)],
    ["NHIF", fmt(data.nhif)],
    ["NSSF (Employee)", fmt(data.nssfEmployee)],
  ];
  if (data.loanDeduction > 0) deductions.push(["Loan Deduction", fmt(data.loanDeduction)]);
  if (data.saccoDeduction > 0) deductions.push(["SACCO Deduction", fmt(data.saccoDeduction)]);
  if (data.otherDeductions > 0) deductions.push(["Other Deductions", fmt(data.otherDeductions)]);
  deductions.push([{ content: "Total Deductions", styles: { fontStyle: "bold" } }, { content: fmt(data.totalDeductions), styles: { fontStyle: "bold" } }]);

  autoTable(doc, {
    startY: afterEarnings,
    head: [["Deductions", "Amount (KES)"]],
    body: deductions,
    theme: "striped",
    headStyles: { fillColor: [140, 40, 40], fontSize: 10 },
    styles: { fontSize: 9 },
    columnStyles: { 1: { halign: "right" } },
  });

  const afterDed = (doc as any).lastAutoTable.finalY + 6;

  // Net pay
  autoTable(doc, {
    startY: afterDed,
    head: [],
    body: [[{ content: "NET PAY", styles: { fontStyle: "bold", fontSize: 12 } }, { content: fmt(data.netPay), styles: { fontStyle: "bold", fontSize: 12, halign: "right" } }]],
    theme: "plain",
    styles: { fillColor: [230, 245, 250] },
  });

  const afterNet = (doc as any).lastAutoTable.finalY + 4;
  doc.setFontSize(8);
  doc.setTextColor(120);
  doc.text(`NSSF Employer Contribution: ${fmt(data.nssfEmployer)}`, 14, afterNet);
  doc.text("This is a computer-generated payslip and does not require a signature.", 14, afterNet + 5);

  doc.save(`payslip-${data.employeeName.replace(/\s+/g, "_")}-${data.payPeriod}.pdf`);
}
