import { useEffect, useState } from "react";
import { supabase } from "@/integrations/supabase/client";
import { useAuth } from "@/contexts/AuthContext";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Users, Calculator, FileText, TrendingUp } from "lucide-react";

export default function DashboardPage() {
  const { isAdmin, isAccountant } = useAuth();
  const [stats, setStats] = useState({
    totalEmployees: 0,
    activeEmployees: 0,
    totalPayroll: 0,
    pendingPayroll: 0,
  });

  useEffect(() => {
    const fetchStats = async () => {
      const [empRes, activeRes, payrollRes, pendingRes] = await Promise.all([
        supabase.from("employees").select("id", { count: "exact", head: true }),
        supabase.from("employees").select("id", { count: "exact", head: true }).eq("is_active", true),
        supabase.from("payroll").select("net_pay").eq("status", "processed"),
        supabase.from("payroll").select("id", { count: "exact", head: true }).eq("status", "draft"),
      ]);

      const totalPay = payrollRes.data?.reduce((sum, p) => sum + Number(p.net_pay), 0) ?? 0;

      setStats({
        totalEmployees: empRes.count ?? 0,
        activeEmployees: activeRes.count ?? 0,
        totalPayroll: totalPay,
        pendingPayroll: pendingRes.count ?? 0,
      });
    };
    fetchStats();
  }, []);

  const statCards = [
    { label: "Total Employees", value: stats.totalEmployees, icon: Users, color: "text-primary" },
    { label: "Active Employees", value: stats.activeEmployees, icon: TrendingUp, color: "text-success" },
    { label: "Total Payroll (KES)", value: `KES ${stats.totalPayroll.toLocaleString()}`, icon: Calculator, color: "text-accent" },
    { label: "Pending Payroll", value: stats.pendingPayroll, icon: FileText, color: "text-warning" },
  ];

  return (
    <div>
      <div className="mb-6">
        <h1 className="text-2xl font-bold">Dashboard</h1>
        <p className="text-muted-foreground text-sm">Payroll overview for Gideon Wheels Ltd</p>
      </div>

      <div className="grid gap-4 sm:grid-cols-2 lg:grid-cols-4">
        {statCards.map((stat) => (
          <Card key={stat.label} className="stat-card">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-muted-foreground">{stat.label}</p>
                  <p className="text-2xl font-bold mt-1">{stat.value}</p>
                </div>
                <stat.icon className={`h-8 w-8 ${stat.color} opacity-80`} />
              </div>
            </CardContent>
          </Card>
        ))}
      </div>
    </div>
  );
}
