import { useEffect, useState } from "react";
import { supabase } from "@/integrations/supabase/client";
import { useAuth } from "@/contexts/AuthContext";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card, CardContent } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Badge } from "@/components/ui/badge";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Plus, Pencil, Search, Loader2, Trash2 } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { logAudit } from "@/lib/audit-logger";

const BRANCHES = [
  "Gideon wheels Ltd. Bamburi",
  "Gideon wheels Ltd. Haile Selassie",
] as const;

interface Employee {
  id: string;
  employee_number: string;
  full_name: string;
  email: string | null;
  phone: string | null;
  department: string;
  designation: string;
  basic_salary: number;
  date_joined: string;
  is_active: boolean;
  payment_method: string;
  branch: string;
}

interface EmployeeSensitiveData {
  employee_id: string;
  kra_pin: string | null;
  nhif_number: string | null;
  nssf_number: string | null;
  bank_name: string | null;
  bank_account: string | null;
  bank_branch: string | null;
  mpesa_number: string | null;
}

const emptyForm = {
  employee_number: "",
  full_name: "",
  email: "",
  phone: "",
  department: "",
  designation: "",
  basic_salary: 0,
  kra_pin: "",
  nhif_number: "",
  nssf_number: "",
  bank_name: "",
  bank_account: "",
  bank_branch: "",
  mpesa_number: "",
  date_joined: new Date().toISOString().split("T")[0],
  payment_method: "bank_transfer",
  branch: BRANCHES[0] as string,
};

export default function EmployeesPage() {
  const [employees, setEmployees] = useState<Employee[]>([]);
  const [sensitiveMap, setSensitiveMap] = useState<Record<string, EmployeeSensitiveData>>({});
  const [loading, setLoading] = useState(true);
  const [search, setSearch] = useState("");
  const [dialogOpen, setDialogOpen] = useState(false);
  const [deleteConfirmId, setDeleteConfirmId] = useState<string | null>(null);
  const [saving, setSaving] = useState(false);
  const [deleting, setDeleting] = useState(false);
  const [editingId, setEditingId] = useState<string | null>(null);
  const [form, setForm] = useState(emptyForm);
  const [activeBranch, setActiveBranch] = useState<string>(BRANCHES[0]);
  const { isAdmin } = useAuth();
  const { toast } = useToast();
  const canEdit = isAdmin();
  const canDelete = isAdmin();

  const fetchEmployees = async () => {
    setLoading(true);
    const [empRes, sensRes] = await Promise.all([
      supabase.from("employees").select("*").order("full_name"),
      supabase.from("employee_sensitive_data").select("*"),
    ]);
    if (empRes.data) setEmployees(empRes.data as Employee[]);
    if (sensRes.data) {
      const map: Record<string, EmployeeSensitiveData> = {};
      for (const s of sensRes.data) {
        map[s.employee_id] = s as EmployeeSensitiveData;
      }
      setSensitiveMap(map);
    }
    setLoading(false);
  };

  useEffect(() => { fetchEmployees(); }, []);

  const handleSave = async (e: React.FormEvent) => {
    e.preventDefault();
    setSaving(true);

    const { kra_pin, nhif_number, nssf_number, bank_name, bank_account, bank_branch, mpesa_number, ...employeePayload } = form;
    const empPayload = { ...employeePayload, basic_salary: Number(employeePayload.basic_salary) };
    const sensPayload = { kra_pin: kra_pin || null, nhif_number: nhif_number || null, nssf_number: nssf_number || null, bank_name: bank_name || null, bank_account: bank_account || null, bank_branch: bank_branch || null, mpesa_number: mpesa_number || null };

    let error;
    if (editingId) {
      const old = employees.find((emp) => emp.id === editingId);
      ({ error } = await supabase.from("employees").update(empPayload).eq("id", editingId));
      if (!error) {
        // Update or insert sensitive data
        const existing = sensitiveMap[editingId];
        if (existing) {
          await supabase.from("employee_sensitive_data").update(sensPayload).eq("employee_id", editingId);
        } else {
          await supabase.from("employee_sensitive_data").insert({ employee_id: editingId, ...sensPayload });
        }
        await logAudit({ action: "employee_updated", tableName: "employees", recordId: editingId, oldData: old as any, newData: { ...empPayload, ...sensPayload } as any });
      }
    } else {
      const { data, error: insertErr } = await supabase.from("employees").insert(empPayload).select("id").single();
      error = insertErr;
      if (!error && data) {
        await supabase.from("employee_sensitive_data").insert({ employee_id: data.id, ...sensPayload });
        await logAudit({ action: "employee_added", tableName: "employees", recordId: data.id, newData: { ...empPayload, ...sensPayload } as any });
      }
    }

    setSaving(false);
    if (error) {
      toast({ title: "Error", description: error.message, variant: "destructive" });
    } else {
      toast({ title: editingId ? "Employee updated" : "Employee added" });
      setDialogOpen(false);
      setEditingId(null);
      setForm(emptyForm);
      fetchEmployees();
    }
  };

  const handleDelete = async (emp: Employee) => {
    setDeleting(true);
    const { error } = await supabase.from("employees").delete().eq("id", emp.id);
    setDeleting(false);
    setDeleteConfirmId(null);
    if (error) {
      toast({ title: "Error", description: error.message, variant: "destructive" });
    } else {
      await logAudit({ action: "employee_deleted", tableName: "employees", recordId: emp.id, oldData: emp as any });
      toast({ title: "Employee deleted", description: `${emp.full_name} has been permanently removed.` });
      fetchEmployees();
    }
  };

  const openEdit = (emp: Employee) => {
    const sens = sensitiveMap[emp.id];
    setEditingId(emp.id);
    setForm({
      employee_number: emp.employee_number,
      full_name: emp.full_name,
      email: emp.email ?? "",
      phone: emp.phone ?? "",
      department: emp.department,
      designation: emp.designation,
      basic_salary: emp.basic_salary,
      kra_pin: sens?.kra_pin ?? "",
      nhif_number: sens?.nhif_number ?? "",
      nssf_number: sens?.nssf_number ?? "",
      bank_name: sens?.bank_name ?? "",
      bank_account: sens?.bank_account ?? "",
      bank_branch: sens?.bank_branch ?? "",
      mpesa_number: sens?.mpesa_number ?? "",
      date_joined: emp.date_joined,
      payment_method: emp.payment_method ?? "bank_transfer",
      branch: emp.branch ?? BRANCHES[0],
    });
    setDialogOpen(true);
  };

  const openNew = () => {
    setEditingId(null);
    setForm({ ...emptyForm, branch: activeBranch });
    setDialogOpen(true);
  };

  const filtered = employees.filter((e) =>
    e.branch === activeBranch &&
    (e.full_name.toLowerCase().includes(search.toLowerCase()) ||
    e.employee_number.toLowerCase().includes(search.toLowerCase()) ||
    e.department.toLowerCase().includes(search.toLowerCase()))
  );

  const updateField = (field: string, value: string | number) =>
    setForm((prev) => ({ ...prev, [field]: value }));

  const deleteTarget = employees.find((e) => e.id === deleteConfirmId);

  const branchCount = (branch: string) => employees.filter((e) => e.branch === branch).length;

  return (
    <div>
      <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4 mb-6">
        <div>
          <h1 className="text-2xl font-bold">Employees</h1>
          <p className="text-muted-foreground text-sm">{employees.length} employees registered</p>
        </div>
        {canEdit && (
          <Button onClick={openNew}>
            <Plus className="h-4 w-4 mr-2" /> Add Employee
          </Button>
        )}
      </div>

      {/* Branch Tabs */}
      <Tabs value={activeBranch} onValueChange={setActiveBranch} className="mb-4">
        <TabsList className="grid w-full grid-cols-2">
          {BRANCHES.map((branch) => (
            <TabsTrigger key={branch} value={branch} className="text-xs sm:text-sm">
              {branch.replace("Gideon wheels Ltd. ", "")} ({branchCount(branch)})
            </TabsTrigger>
          ))}
        </TabsList>
      </Tabs>

      {/* Search */}
      <div className="relative mb-4 max-w-sm">
        <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
        <Input placeholder="Search employees..." className="pl-9" value={search} onChange={(e) => setSearch(e.target.value)} />
      </div>

      {/* Table */}
      <Card>
        <CardContent className="p-0">
          <div className="overflow-x-auto">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Employee #</TableHead>
                  <TableHead>Name</TableHead>
                  <TableHead>Department</TableHead>
                  <TableHead>Designation</TableHead>
                  <TableHead>Payment</TableHead>
                  <TableHead className="text-right">Basic Salary</TableHead>
                  <TableHead>Status</TableHead>
                  {canEdit && <TableHead>Actions</TableHead>}
                </TableRow>
              </TableHeader>
              <TableBody>
                {loading ? (
                  <TableRow><TableCell colSpan={8} className="text-center py-8"><Loader2 className="h-5 w-5 animate-spin mx-auto" /></TableCell></TableRow>
                ) : filtered.length === 0 ? (
                  <TableRow><TableCell colSpan={8} className="text-center py-8 text-muted-foreground">No employees found in this branch</TableCell></TableRow>
                ) : (
                  filtered.map((emp) => (
                    <TableRow key={emp.id}>
                      <TableCell className="font-mono text-sm">{emp.employee_number}</TableCell>
                      <TableCell className="font-medium">{emp.full_name}</TableCell>
                      <TableCell>{emp.department}</TableCell>
                      <TableCell>{emp.designation}</TableCell>
                      <TableCell>
                        <Badge variant="outline" className="text-xs">
                          {emp.payment_method === "mpesa" ? "📱 M-Pesa" : "🏦 Bank"}
                        </Badge>
                      </TableCell>
                      <TableCell className="text-right font-mono">{Number(emp.basic_salary).toLocaleString()}</TableCell>
                      <TableCell>
                        <Badge variant={emp.is_active ? "default" : "secondary"}>
                          {emp.is_active ? "Active" : "Inactive"}
                        </Badge>
                      </TableCell>
                      {canEdit && (
                        <TableCell>
                          <div className="flex gap-1">
                            <Button variant="ghost" size="sm" onClick={() => openEdit(emp)}>
                              <Pencil className="h-3.5 w-3.5" />
                            </Button>
                            {canDelete && (
                              <Button variant="ghost" size="sm" className="text-destructive hover:text-destructive" onClick={() => setDeleteConfirmId(emp.id)}>
                                <Trash2 className="h-3.5 w-3.5" />
                              </Button>
                            )}
                          </div>
                        </TableCell>
                      )}
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          </div>
        </CardContent>
      </Card>

      {/* Add/Edit Dialog */}
      <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
        <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>{editingId ? "Edit Employee" : "Add Employee"}</DialogTitle>
          </DialogHeader>
          <form onSubmit={handleSave} className="space-y-4">
            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label>Branch *</Label>
                <Select value={form.branch} onValueChange={(v) => updateField("branch", v)}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {BRANCHES.map((b) => (
                      <SelectItem key={b} value={b}>{b}</SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>Employee Number *</Label>
                <Input value={form.employee_number} onChange={(e) => updateField("employee_number", e.target.value)} required />
              </div>
              <div className="space-y-2">
                <Label>Full Name *</Label>
                <Input value={form.full_name} onChange={(e) => updateField("full_name", e.target.value)} required />
              </div>
              <div className="space-y-2">
                <Label>Email</Label>
                <Input type="email" value={form.email} onChange={(e) => updateField("email", e.target.value)} />
              </div>
              <div className="space-y-2">
                <Label>Phone</Label>
                <Input value={form.phone} onChange={(e) => updateField("phone", e.target.value)} />
              </div>
              <div className="space-y-2">
                <Label>Department *</Label>
                <Input value={form.department} onChange={(e) => updateField("department", e.target.value)} required />
              </div>
              <div className="space-y-2">
                <Label>Designation *</Label>
                <Input value={form.designation} onChange={(e) => updateField("designation", e.target.value)} required />
              </div>
              <div className="space-y-2">
                <Label>Basic Salary (KES) *</Label>
                <Input type="number" min={0} value={form.basic_salary} onChange={(e) => updateField("basic_salary", e.target.value)} required />
              </div>
              <div className="space-y-2">
                <Label>Date Joined</Label>
                <Input type="date" value={form.date_joined} onChange={(e) => updateField("date_joined", e.target.value)} />
              </div>
              <div className="space-y-2">
                <Label>KRA PIN</Label>
                <Input value={form.kra_pin} onChange={(e) => updateField("kra_pin", e.target.value)} placeholder="A00XXXXXXXB" />
              </div>
              <div className="space-y-2">
                <Label>SHA Number</Label>
                <Input value={form.nhif_number} onChange={(e) => updateField("nhif_number", e.target.value)} placeholder="SHA membership number" />
              </div>
              <div className="space-y-2">
                <Label>NSSF Number</Label>
                <Input value={form.nssf_number} onChange={(e) => updateField("nssf_number", e.target.value)} />
              </div>
            </div>

            {/* Payment Method */}
            <div className="border-t pt-4">
              <p className="text-sm font-medium mb-3">Payment Method</p>
              <div className="space-y-2">
                <Label>Preferred Payment Method</Label>
                <Select value={form.payment_method} onValueChange={(v) => updateField("payment_method", v)}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="bank_transfer">🏦 Bank Transfer</SelectItem>
                    <SelectItem value="mpesa">📱 M-Pesa</SelectItem>
                    <SelectItem value="cash">💵 Cash</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              {form.payment_method === "mpesa" ? (
                <div className="space-y-2 mt-3">
                  <Label>M-Pesa Number *</Label>
                  <Input
                    value={form.mpesa_number}
                    onChange={(e) => updateField("mpesa_number", e.target.value)}
                    placeholder="07XXXXXXXX or 254XXXXXXXXX"
                    required={form.payment_method === "mpesa"}
                  />
                </div>
              ) : (
                <div className="grid grid-cols-1 sm:grid-cols-3 gap-4 mt-3">
                  <div className="space-y-2">
                    <Label>Bank Name</Label>
                    <Input value={form.bank_name} onChange={(e) => updateField("bank_name", e.target.value)} />
                  </div>
                  <div className="space-y-2">
                    <Label>Account Number</Label>
                    <Input value={form.bank_account} onChange={(e) => updateField("bank_account", e.target.value)} />
                  </div>
                  <div className="space-y-2">
                    <Label>Branch</Label>
                    <Input value={form.bank_branch} onChange={(e) => updateField("bank_branch", e.target.value)} />
                  </div>
                </div>
              )}
            </div>

            <div className="flex justify-end gap-2 pt-2">
              <Button type="button" variant="outline" onClick={() => setDialogOpen(false)}>Cancel</Button>
              <Button type="submit" disabled={saving}>
                {saving && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
                {editingId ? "Update" : "Add"} Employee
              </Button>
            </div>
          </form>
        </DialogContent>
      </Dialog>

      {/* Delete Confirm Dialog */}
      <Dialog open={!!deleteConfirmId} onOpenChange={() => setDeleteConfirmId(null)}>
        <DialogContent className="max-w-sm">
          <DialogHeader>
            <DialogTitle>Delete Employee Permanently</DialogTitle>
          </DialogHeader>
          <p className="text-sm text-muted-foreground">
            Are you sure you want to <strong>permanently delete</strong> <strong>{deleteTarget?.full_name}</strong>? This action cannot be undone and all their records will be removed.
          </p>
          <div className="flex justify-end gap-2 pt-2">
            <Button variant="outline" onClick={() => setDeleteConfirmId(null)}>Cancel</Button>
            <Button variant="destructive" disabled={deleting} onClick={() => deleteTarget && handleDelete(deleteTarget)}>
              {deleting && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
              Delete Permanently
            </Button>
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
}
