import { useEffect, useState } from "react";
import { supabase } from "@/integrations/supabase/client";
import { useAuth } from "@/contexts/AuthContext";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card, CardContent } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { Calculator, Loader2, Check, Users, Pencil, Undo2, Trash2 } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { computePayroll } from "@/lib/payroll-calculations";
import { logAudit } from "@/lib/audit-logger";

interface Employee {
  id: string;
  employee_number: string;
  full_name: string;
  department: string;
  basic_salary: number;
}

interface PayrollRecord {
  id: string;
  employee_id: string;
  pay_period: string;
  basic_salary: number;
  housing_allowance: number;
  transport_allowance: number;
  other_allowances: number;
  overtime_pay: number;
  bonus: number;
  gross_pay: number;
  paye: number;
  nhif: number;
  nssf_employee: number;
  nssf_employer: number;
  housing_levy_employee: number;
  housing_levy_employer: number;
  loan_deduction: number;
  sacco_deduction: number;
  other_deductions: number;
  total_deductions: number;
  net_pay: number;
  status: string;
  employees?: Employee;
}

type AllowanceFieldsData = { housing: string; transport: string; other: string; overtime: string; bonus: string; loan: string; sacco: string; otherDed: string };

const AllowanceFieldsInputs = ({ values, onChange }: { values: AllowanceFieldsData; onChange: (v: AllowanceFieldsData) => void }) => (
  <>
    <div className="border-t pt-4">
      <p className="text-sm font-medium mb-3">Allowances & Earnings</p>
      <div className="grid grid-cols-2 gap-3">
        {[{ key: "housing", label: "Housing" }, { key: "transport", label: "Transport" }, { key: "overtime", label: "Overtime" }, { key: "bonus", label: "Bonus" }]
          .map(({ key, label }) => (
            <div key={key} className="space-y-1">
              <Label className="text-xs">{label}</Label>
              <Input
                type="text"
                inputMode="decimal"
                placeholder="0"
                value={(values as any)[key]}
                onChange={(e) => onChange({ ...values, [key]: e.target.value })}
              />
            </div>
          ))}
        <div className="space-y-1 col-span-2">
          <Label className="text-xs">Other Allowances</Label>
          <Input
            type="text"
            inputMode="decimal"
            placeholder="0"
            value={values.other}
            onChange={(e) => onChange({ ...values, other: e.target.value })}
          />
        </div>
      </div>
    </div>
    <div className="border-t pt-4">
      <p className="text-sm font-medium mb-3">Deductions (Non-statutory)</p>
      <div className="grid grid-cols-2 gap-3">
        {[{ key: "loan", label: "Loan" }, { key: "sacco", label: "SACCO" }].map(({ key, label }) => (
          <div key={key} className="space-y-1">
            <Label className="text-xs">{label}</Label>
            <Input
              type="text"
              inputMode="decimal"
              placeholder="0"
              value={(values as any)[key]}
              onChange={(e) => onChange({ ...values, [key]: e.target.value })}
            />
          </div>
        ))}
        <div className="space-y-1 col-span-2">
          <Label className="text-xs">Other Deductions</Label>
          <Input
            type="text"
            inputMode="decimal"
            placeholder="0"
            value={values.otherDed}
            onChange={(e) => onChange({ ...values, otherDed: e.target.value })}
          />
        </div>
      </div>
    </div>
    <p className="text-xs text-muted-foreground">PAYE, SHA (2.75%), NSSF, and Housing Levy (1.5%) are calculated automatically per Kenyan statutory rates.</p>
  </>
);

export default function PayrollPage() {
  const [employees, setEmployees] = useState<Employee[]>([]);
  const [payrolls, setPayrolls] = useState<PayrollRecord[]>([]);
  const [loading, setLoading] = useState(true);
  const [processing, setProcessing] = useState(false);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [batchDialogOpen, setBatchDialogOpen] = useState(false);
  const [editDialogOpen, setEditDialogOpen] = useState(false);
  const [deleteConfirmId, setDeleteConfirmId] = useState<string | null>(null);
  const [editingPayroll, setEditingPayroll] = useState<PayrollRecord | null>(null);
  const [selectedEmployee, setSelectedEmployee] = useState("");
  const [payPeriod, setPayPeriod] = useState(new Date().toISOString().slice(0, 7));
  const [batchPeriod, setBatchPeriod] = useState(new Date().toISOString().slice(0, 7));
  const [batchProgress, setBatchProgress] = useState({ done: 0, total: 0, errors: 0 });
  const { isAdmin, isAccountant, user } = useAuth();
  const { toast } = useToast();
  const canProcess = isAdmin() || isAccountant();
  const canDelete = isAdmin();

  const defaultAllowances: AllowanceFieldsData = { housing: "", transport: "", other: "", overtime: "", bonus: "", loan: "", sacco: "", otherDed: "" };
  const [allowances, setAllowances] = useState<AllowanceFieldsData>(defaultAllowances);
  const [editAllowances, setEditAllowances] = useState<AllowanceFieldsData>(defaultAllowances);

  const parseAllowances = (a: AllowanceFieldsData) => ({
    housing: parseFloat(a.housing) || 0,
    transport: parseFloat(a.transport) || 0,
    other: parseFloat(a.other) || 0,
    overtime: parseFloat(a.overtime) || 0,
    bonus: parseFloat(a.bonus) || 0,
    loan: parseFloat(a.loan) || 0,
    sacco: parseFloat(a.sacco) || 0,
    otherDed: parseFloat(a.otherDed) || 0,
  });

  const fetchData = async () => {
    setLoading(true);
    const [empRes, payRes] = await Promise.all([
      supabase.from("employees").select("id, employee_number, full_name, department, basic_salary").eq("is_active", true).order("full_name"),
      supabase.from("payroll").select("*, employees(employee_number, full_name, department)").order("processed_at", { ascending: false }).limit(100),
    ]);
    if (empRes.data) setEmployees(empRes.data);
    if (payRes.data) setPayrolls(payRes.data as any);
    setLoading(false);
  };

  useEffect(() => { fetchData(); }, []);

  const buildRecord = (emp: Employee, a: AllowanceFieldsData) => {
    const p = parseAllowances(a);
    const result = computePayroll({
      basicSalary: Number(emp.basic_salary),
      housing: p.housing, transport: p.transport,
      other: p.other, overtime: p.overtime, bonus: p.bonus,
      loan: p.loan, sacco: p.sacco, otherDed: p.otherDed,
    });
    return { result, record: {
      employee_id: emp.id,
      basic_salary: emp.basic_salary,
      housing_allowance: p.housing,
      transport_allowance: p.transport,
      other_allowances: p.other,
      overtime_pay: p.overtime,
      bonus: p.bonus,
      gross_pay: result.grossPay,
      paye: result.paye,
      nhif: result.sha,
      nssf_employee: result.nssf.employee,
      nssf_employer: result.nssf.employer,
      housing_levy_employee: result.housingLevy.employee,
      housing_levy_employer: result.housingLevy.employer,
      loan_deduction: p.loan,
      sacco_deduction: p.sacco,
      other_deductions: p.otherDed,
      total_deductions: result.totalDeductions,
      net_pay: result.netPay,
      processed_by: user?.id,
      status: "processed",
    }};
  };

  const processPayroll = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!selectedEmployee || !payPeriod) return;
    setProcessing(true);
    const emp = employees.find((e) => e.id === selectedEmployee);
    if (!emp) { setProcessing(false); return; }

    const { result, record } = buildRecord(emp, allowances);
    const { data, error } = await supabase.from("payroll").insert({ ...record, pay_period: payPeriod }).select("id").single();

    setProcessing(false);
    if (error) {
      toast({ title: "Error", description: error.message.includes("unique") ? "Payroll already processed for this period" : error.message, variant: "destructive" });
    } else {
      await logAudit({ action: "payroll_processed", tableName: "payroll", recordId: data?.id, newData: record as any });
      toast({ title: "Payroll processed", description: `Net pay: KES ${result.netPay.toLocaleString()}` });
      setDialogOpen(false);
      setAllowances(defaultAllowances);
      fetchData();
    }
  };

  const processBatchPayroll = async () => {
    if (!batchPeriod) return;
    setProcessing(true);
    setBatchProgress({ done: 0, total: employees.length, errors: 0 });
    let done = 0; let errors = 0;

    for (const emp of employees) {
      const { result, record } = buildRecord(emp, defaultAllowances);
      const { data, error } = await supabase.from("payroll").insert({ ...record, pay_period: batchPeriod }).select("id").single();
      if (error) { errors++; } else {
        await logAudit({ action: "batch_payroll_processed", tableName: "payroll", recordId: data?.id, newData: record as any });
      }
      done++;
      setBatchProgress({ done, total: employees.length, errors });
    }

    setProcessing(false);
    toast({ title: "Batch payroll complete", description: `Processed ${done - errors}/${done} employees. ${errors > 0 ? `${errors} skipped.` : ""}` });
    setBatchDialogOpen(false);
    fetchData();
  };

  const openEditDialog = (p: PayrollRecord) => {
    setEditingPayroll(p);
    setEditAllowances({
      housing: String(p.housing_allowance || ""), transport: String(p.transport_allowance || ""),
      other: String(p.other_allowances || ""), overtime: String(p.overtime_pay || ""), bonus: String(p.bonus || ""),
      loan: String(p.loan_deduction || ""), sacco: String(p.sacco_deduction || ""), otherDed: String(p.other_deductions || ""),
    });
    setEditDialogOpen(true);
  };

  const saveEditPayroll = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!editingPayroll) return;
    setProcessing(true);

    const emp = { id: editingPayroll.employee_id, employee_number: "", full_name: "", department: "", basic_salary: editingPayroll.basic_salary };
    const { result, record } = buildRecord(emp as Employee, editAllowances);

    const oldData = {
      housing_allowance: editingPayroll.housing_allowance, transport_allowance: editingPayroll.transport_allowance,
      other_allowances: editingPayroll.other_allowances, overtime_pay: editingPayroll.overtime_pay,
      bonus: editingPayroll.bonus, gross_pay: editingPayroll.gross_pay,
      paye: editingPayroll.paye, nhif: editingPayroll.nhif,
      total_deductions: editingPayroll.total_deductions, net_pay: editingPayroll.net_pay,
    };

    const parsed = parseAllowances(editAllowances);
    const updates = {
      housing_allowance: parsed.housing, transport_allowance: parsed.transport,
      other_allowances: parsed.other, overtime_pay: parsed.overtime,
      bonus: parsed.bonus, gross_pay: result.grossPay,
      paye: result.paye, nhif: result.sha,
      nssf_employee: result.nssf.employee, nssf_employer: result.nssf.employer,
      housing_levy_employee: result.housingLevy.employee, housing_levy_employer: result.housingLevy.employer,
      loan_deduction: parsed.loan, sacco_deduction: parsed.sacco,
      other_deductions: parsed.otherDed,
      total_deductions: result.totalDeductions, net_pay: result.netPay,
    };

    const { error } = await supabase.from("payroll").update(updates).eq("id", editingPayroll.id);

    if (error) {
      setProcessing(false);
      toast({ title: "Error", description: error.message, variant: "destructive" });
    } else {
      await logAudit({ action: "payroll_edited", tableName: "payroll", recordId: editingPayroll.id, oldData: oldData as any, newData: updates as any });
      setEditDialogOpen(false);
      setEditingPayroll(null);
      await fetchData();
      setProcessing(false);
      toast({ title: "Payroll updated", description: `New net pay: KES ${result.netPay.toLocaleString()}` });
    }
  };

  const reversePayroll = async (p: PayrollRecord) => {
    if (!confirm(`Reverse payroll for ${(p as any).employees?.full_name ?? "this employee"} (${p.pay_period})? This cannot be undone.`)) return;
    const { error } = await supabase.from("payroll").update({ status: "reversed" }).eq("id", p.id);
    if (error) {
      toast({ title: "Error", description: error.message, variant: "destructive" });
    } else {
      await logAudit({ action: "payroll_reversed", tableName: "payroll", recordId: p.id, oldData: { status: p.status } as any, newData: { status: "reversed" } as any });
      toast({ title: "Payroll reversed" });
      fetchData();
    }
  };

  const deletePayroll = async (p: PayrollRecord) => {
    const { error } = await supabase.from("payroll").delete().eq("id", p.id);
    setDeleteConfirmId(null);
    if (error) {
      toast({ title: "Error", description: error.message, variant: "destructive" });
    } else {
      await logAudit({ action: "payroll_deleted", tableName: "payroll", recordId: p.id, oldData: p as any });
      toast({ title: "Record deleted", description: `Payroll for ${(p as any).employees?.full_name} (${p.pay_period}) permanently deleted.` });
      fetchData();
    }
  };

  const statusBadge = (status: string) => {
    const variants: Record<string, "default" | "secondary" | "destructive"> = { processed: "default", reversed: "destructive", draft: "secondary" };
    return (
      <Badge variant={variants[status] ?? "secondary"}>
        {status === "processed" && <Check className="h-3 w-3 mr-1" />}
        {status === "reversed" && <Undo2 className="h-3 w-3 mr-1" />}
        {status}
      </Badge>
    );
  };


  const deleteTarget = payrolls.find((p) => p.id === deleteConfirmId);

  return (
    <div>
      <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4 mb-6">
        <div>
          <h1 className="text-2xl font-bold">Payroll</h1>
          <p className="text-muted-foreground text-sm">Process and manage monthly payroll</p>
        </div>
        {canProcess && (
          <div className="flex gap-2">
            <Button variant="outline" onClick={() => setBatchDialogOpen(true)}><Users className="h-4 w-4 mr-2" /> Batch Process</Button>
            <Button onClick={() => setDialogOpen(true)}><Calculator className="h-4 w-4 mr-2" /> Process Payroll</Button>
          </div>
        )}
      </div>

      <Card>
        <CardContent className="p-0">
          <div className="overflow-x-auto">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Period</TableHead>
                  <TableHead>Employee</TableHead>
                  <TableHead>Department</TableHead>
                  <TableHead className="text-right">Gross Pay</TableHead>
                  <TableHead className="text-right">Deductions</TableHead>
                  <TableHead className="text-right">Net Pay</TableHead>
                  <TableHead>Status</TableHead>
                  {canProcess && <TableHead>Actions</TableHead>}
                </TableRow>
              </TableHeader>
              <TableBody>
                {loading ? (
                  <TableRow><TableCell colSpan={8} className="text-center py-8"><Loader2 className="h-5 w-5 animate-spin mx-auto" /></TableCell></TableRow>
                ) : payrolls.length === 0 ? (
                  <TableRow><TableCell colSpan={8} className="text-center py-8 text-muted-foreground">No payroll records</TableCell></TableRow>
                ) : (
                  payrolls.map((p) => (
                    <TableRow key={p.id} className={p.status === "reversed" ? "opacity-60" : ""}>
                      <TableCell className="font-mono text-sm">{p.pay_period}</TableCell>
                      <TableCell className="font-medium">{(p as any).employees?.full_name ?? "—"}</TableCell>
                      <TableCell>{(p as any).employees?.department ?? "—"}</TableCell>
                      <TableCell className="text-right font-mono">{Number(p.gross_pay).toLocaleString()}</TableCell>
                      <TableCell className="text-right font-mono">{Number(p.total_deductions).toLocaleString()}</TableCell>
                      <TableCell className="text-right font-mono font-semibold">{Number(p.net_pay).toLocaleString()}</TableCell>
                      <TableCell>{statusBadge(p.status)}</TableCell>
                      {canProcess && (
                        <TableCell>
                          <div className="flex gap-1">
                            {p.status === "processed" && (
                              <Button variant="ghost" size="sm" onClick={() => openEditDialog(p)}>
                                <Pencil className="h-3.5 w-3.5" />
                              </Button>
                            )}
                            {p.status === "processed" && (
                              <Button variant="ghost" size="sm" className="text-warning hover:text-warning/80" onClick={() => reversePayroll(p)}>
                                <Undo2 className="h-3.5 w-3.5" />
                              </Button>
                            )}
                            {canDelete && (
                              <Button variant="ghost" size="sm" className="text-destructive hover:text-destructive" onClick={() => setDeleteConfirmId(p.id)}>
                                <Trash2 className="h-3.5 w-3.5" />
                              </Button>
                            )}
                          </div>
                        </TableCell>
                      )}
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          </div>
        </CardContent>
      </Card>

      {/* Single process dialog */}
      <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
        <DialogContent className="max-w-lg max-h-[90vh] overflow-y-auto">
          <DialogHeader><DialogTitle>Process Payroll</DialogTitle></DialogHeader>
          <form onSubmit={processPayroll} className="space-y-4">
            <div className="space-y-2">
              <Label>Pay Period</Label>
              <Input type="month" value={payPeriod} onChange={(e) => setPayPeriod(e.target.value)} required />
            </div>
            <div className="space-y-2">
              <Label>Employee</Label>
              <Select value={selectedEmployee} onValueChange={setSelectedEmployee}>
                <SelectTrigger><SelectValue placeholder="Select employee" /></SelectTrigger>
                <SelectContent>
                  {employees.map((e) => (
                    <SelectItem key={e.id} value={e.id}>
                      {e.full_name} ({e.employee_number}) — KES {Number(e.basic_salary).toLocaleString()}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <AllowanceFieldsInputs values={allowances} onChange={setAllowances} />
            <div className="flex justify-end gap-2 pt-2">
              <Button type="button" variant="outline" onClick={() => setDialogOpen(false)}>Cancel</Button>
              <Button type="submit" disabled={processing || !selectedEmployee}>
                {processing && <Loader2 className="mr-2 h-4 w-4 animate-spin" />} Process
              </Button>
            </div>
          </form>
        </DialogContent>
      </Dialog>

      {/* Edit payroll dialog */}
      <Dialog open={editDialogOpen} onOpenChange={setEditDialogOpen}>
        <DialogContent className="max-w-lg max-h-[90vh] overflow-y-auto">
          <DialogHeader><DialogTitle>Edit Payroll — {editingPayroll?.pay_period}</DialogTitle></DialogHeader>
          {editingPayroll && (
            <form onSubmit={saveEditPayroll} className="space-y-4">
              <div className="bg-muted p-3 rounded-md text-sm">
                <span className="font-medium">{(editingPayroll as any).employees?.full_name}</span>
                <span className="text-muted-foreground"> — Basic: KES {Number(editingPayroll.basic_salary).toLocaleString()}</span>
              </div>
              <AllowanceFieldsInputs values={editAllowances} onChange={setEditAllowances} />
              <div className="flex justify-end gap-2 pt-2">
                <Button type="button" variant="outline" onClick={() => setEditDialogOpen(false)}>Cancel</Button>
                <Button type="submit" disabled={processing}>
                  {processing && <Loader2 className="mr-2 h-4 w-4 animate-spin" />} Save Changes
                </Button>
              </div>
            </form>
          )}
        </DialogContent>
      </Dialog>

      {/* Batch process dialog */}
      <Dialog open={batchDialogOpen} onOpenChange={setBatchDialogOpen}>
        <DialogContent className="max-w-md">
          <DialogHeader><DialogTitle>Batch Process Payroll</DialogTitle></DialogHeader>
          <div className="space-y-4">
            <p className="text-sm text-muted-foreground">
              Process payroll for all <strong>{employees.length}</strong> active employees using their basic salary only.
            </p>
            <div className="space-y-2">
              <Label>Pay Period</Label>
              <Input type="month" value={batchPeriod} onChange={(e) => setBatchPeriod(e.target.value)} />
            </div>
            {processing && (
              <div className="space-y-2">
                <div className="flex justify-between text-sm"><span>Progress</span><span>{batchProgress.done}/{batchProgress.total}</span></div>
                <div className="w-full bg-muted rounded-full h-2">
                  <div className="bg-primary h-2 rounded-full transition-all" style={{ width: `${(batchProgress.done / Math.max(batchProgress.total, 1)) * 100}%` }} />
                </div>
                {batchProgress.errors > 0 && <p className="text-xs text-destructive">{batchProgress.errors} errors</p>}
              </div>
            )}
            <div className="flex justify-end gap-2">
              <Button variant="outline" onClick={() => setBatchDialogOpen(false)} disabled={processing}>Cancel</Button>
              <Button onClick={processBatchPayroll} disabled={processing || employees.length === 0}>
                {processing ? <><Loader2 className="mr-2 h-4 w-4 animate-spin" /> Processing...</> : <>Process All</>}
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>

      {/* Delete confirm dialog */}
      <Dialog open={!!deleteConfirmId} onOpenChange={() => setDeleteConfirmId(null)}>
        <DialogContent className="max-w-sm">
          <DialogHeader><DialogTitle>Delete Payroll Record</DialogTitle></DialogHeader>
          <p className="text-sm text-muted-foreground">
            Permanently delete payroll record for <strong>{(deleteTarget as any)?.employees?.full_name}</strong> ({deleteTarget?.pay_period})?
            This action <strong>cannot be undone</strong>.
          </p>
          <div className="flex justify-end gap-2 pt-2">
            <Button variant="outline" onClick={() => setDeleteConfirmId(null)}>Cancel</Button>
            <Button variant="destructive" onClick={() => deleteTarget && deletePayroll(deleteTarget)}>
              Delete Permanently
            </Button>
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
}
