import { useEffect, useState } from "react";
import { supabase } from "@/integrations/supabase/client";
import { Card, CardContent } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Input } from "@/components/ui/input";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Search, Loader2, FileText, Printer, Download, Mail } from "lucide-react";
import { Separator } from "@/components/ui/separator";
import { generatePayslipPDF } from "@/lib/payslip-pdf";
import { useToast } from "@/hooks/use-toast";

interface PayslipData {
  id: string;
  pay_period: string;
  employee_id: string;
  basic_salary: number;
  housing_allowance: number;
  transport_allowance: number;
  other_allowances: number;
  overtime_pay: number;
  bonus: number;
  gross_pay: number;
  paye: number;
  nhif: number;
  nssf_employee: number;
  nssf_employer: number;
  housing_levy_employee: number;
  housing_levy_employer: number;
  loan_deduction: number;
  sacco_deduction: number;
  other_deductions: number;
  total_deductions: number;
  net_pay: number;
  status: string;
  processed_at: string;
  employees: {
    employee_number: string;
    full_name: string;
    department: string;
    designation: string;
    email: string | null;
  };
}

interface SensitiveInfo {
  kra_pin: string | null;
  nhif_number: string | null;
  nssf_number: string | null;
  bank_name: string | null;
  bank_account: string | null;
  bank_branch: string | null;
  mpesa_number: string | null;
}

const fmt = (n: number) => Number(n).toLocaleString("en-KE", { minimumFractionDigits: 2 });

export default function PayslipsPage() {
  const [payslips, setPayslips] = useState<PayslipData[]>([]);
  const [loading, setLoading] = useState(true);
  const [search, setSearch] = useState("");
  const [selected, setSelected] = useState<PayslipData | null>(null);
  const [selectedSensitive, setSelectedSensitive] = useState<SensitiveInfo | null>(null);
  const [sendingEmail, setSendingEmail] = useState<string | null>(null);
  const { toast } = useToast();

  useEffect(() => {
    const fetch = async () => {
      setLoading(true);
      const { data } = await supabase
        .from("payroll")
        .select("*, employees(employee_number, full_name, department, designation, email)")
        .eq("status", "processed")
        .order("processed_at", { ascending: false });
      if (data) setPayslips(data as any);
      setLoading(false);
    };
    fetch();
  }, []);

  const fetchSensitiveInfo = async (employeeId: string): Promise<SensitiveInfo> => {
    const { data } = await supabase.rpc("get_employee_sensitive_info", { p_employee_id: employeeId });
    if (data && data.length > 0) return data[0] as SensitiveInfo;
    return { kra_pin: null, nhif_number: null, nssf_number: null, bank_name: null, bank_account: null, bank_branch: null, mpesa_number: null };
  };

  const handleViewPayslip = async (p: PayslipData) => {
    setSelected(p);
    const sens = await fetchSensitiveInfo(p.employee_id);
    setSelectedSensitive(sens);
  };

  const filtered = payslips.filter((p) =>
    (p.employees?.full_name ?? "").toLowerCase().includes(search.toLowerCase()) ||
    p.pay_period.includes(search)
  );

  const handlePrint = () => window.print();

  const handleDownloadPDF = async (p: PayslipData) => {
    const sens = await fetchSensitiveInfo(p.employee_id);
    generatePayslipPDF({
      payPeriod: p.pay_period,
      employeeName: p.employees?.full_name ?? "",
      employeeNumber: p.employees?.employee_number ?? "",
      department: p.employees?.department ?? "",
      designation: p.employees?.designation ?? "",
      kraPin: sens.kra_pin ?? "—",
      nhifNumber: sens.nhif_number ?? "—",
      nssfNumber: sens.nssf_number ?? "—",
      bankName: sens.bank_name ?? "—",
      bankAccount: sens.bank_account ?? "—",
      basicSalary: Number(p.basic_salary),
      housingAllowance: Number(p.housing_allowance),
      transportAllowance: Number(p.transport_allowance),
      otherAllowances: Number(p.other_allowances),
      overtimePay: Number(p.overtime_pay),
      bonus: Number(p.bonus),
      grossPay: Number(p.gross_pay),
      paye: Number(p.paye),
      nhif: Number(p.nhif),
      nssfEmployee: Number(p.nssf_employee),
      nssfEmployer: Number(p.nssf_employer),
      loanDeduction: Number(p.loan_deduction),
      saccoDeduction: Number(p.sacco_deduction),
      otherDeductions: Number(p.other_deductions),
      totalDeductions: Number(p.total_deductions),
      netPay: Number(p.net_pay),
    });
  };

  const handleEmailPayslip = async (p: PayslipData) => {
    const employeeEmail = p.employees?.email;
    if (!employeeEmail) {
      toast({ title: "No email", description: "This employee has no email address on file.", variant: "destructive" });
      return;
    }

    setSendingEmail(p.id);
    try {
      const res = await supabase.functions.invoke("send-payslip", {
        body: { payrollId: p.id, recipientEmail: employeeEmail },
      });

      if (res.error) throw res.error;
      toast({ title: "Email sent", description: `Payslip emailed to ${employeeEmail}` });
    } catch (err: any) {
      toast({ title: "Email failed", description: err.message || "Could not send email", variant: "destructive" });
    } finally {
      setSendingEmail(null);
    }
  };

  return (
    <div>
      <div className="mb-6">
        <h1 className="text-2xl font-bold">Payslips</h1>
        <p className="text-muted-foreground text-sm">View, print, download, and email employee payslips</p>
      </div>

      <div className="relative mb-4 max-w-sm">
        <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
        <Input placeholder="Search by name or period..." className="pl-9" value={search} onChange={(e) => setSearch(e.target.value)} />
      </div>

      <Card>
        <CardContent className="p-0">
          <div className="overflow-x-auto">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Period</TableHead>
                  <TableHead>Employee</TableHead>
                  <TableHead>Department</TableHead>
                  <TableHead className="text-right">Net Pay</TableHead>
                  <TableHead>Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {loading ? (
                  <TableRow><TableCell colSpan={5} className="text-center py-8"><Loader2 className="h-5 w-5 animate-spin mx-auto" /></TableCell></TableRow>
                ) : filtered.length === 0 ? (
                  <TableRow><TableCell colSpan={5} className="text-center py-8 text-muted-foreground">No payslips found</TableCell></TableRow>
                ) : (
                  filtered.map((p) => (
                    <TableRow key={p.id}>
                      <TableCell className="font-mono text-sm">{p.pay_period}</TableCell>
                      <TableCell className="font-medium">{p.employees?.full_name}</TableCell>
                      <TableCell>{p.employees?.department}</TableCell>
                      <TableCell className="text-right font-mono font-semibold">KES {fmt(p.net_pay)}</TableCell>
                      <TableCell>
                        <div className="flex gap-1">
                          <Button variant="ghost" size="sm" onClick={() => handleViewPayslip(p)}>
                            <FileText className="h-3.5 w-3.5 mr-1" /> View
                          </Button>
                          <Button variant="ghost" size="sm" onClick={() => handleDownloadPDF(p)}>
                            <Download className="h-3.5 w-3.5 mr-1" /> PDF
                          </Button>
                          <Button variant="ghost" size="sm" onClick={() => handleEmailPayslip(p)} disabled={sendingEmail === p.id}>
                            {sendingEmail === p.id ? <Loader2 className="h-3.5 w-3.5 mr-1 animate-spin" /> : <Mail className="h-3.5 w-3.5 mr-1" />}
                            Email
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          </div>
        </CardContent>
      </Card>

      {/* Payslip dialog */}
      <Dialog open={!!selected} onOpenChange={() => { setSelected(null); setSelectedSensitive(null); }}>
        <DialogContent className="max-w-xl max-h-[90vh] overflow-y-auto print:max-w-none print:shadow-none">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              Payslip — {selected?.pay_period}
              <div className="ml-auto flex gap-1 print:hidden">
                <Button variant="outline" size="sm" onClick={() => selected && handleDownloadPDF(selected)}>
                  <Download className="h-3.5 w-3.5 mr-1" /> PDF
                </Button>
                <Button variant="outline" size="sm" onClick={() => selected && handleEmailPayslip(selected)} disabled={!!sendingEmail}>
                  {sendingEmail ? <Loader2 className="h-3.5 w-3.5 mr-1 animate-spin" /> : <Mail className="h-3.5 w-3.5 mr-1" />}
                  Email
                </Button>
                <Button variant="outline" size="sm" onClick={handlePrint}>
                  <Printer className="h-3.5 w-3.5 mr-1" /> Print
                </Button>
              </div>
            </DialogTitle>
          </DialogHeader>

          {selected && (
            <div className="space-y-4 text-sm">
              <div className="text-center border-b pb-4">
                <h2 className="text-lg font-bold">GIDEON WHEELS LTD</h2>
                <p className="text-muted-foreground">Payslip for {selected.pay_period}</p>
              </div>
              <div className="grid grid-cols-2 gap-2">
                <div><span className="text-muted-foreground">Name:</span> {selected.employees?.full_name}</div>
                <div><span className="text-muted-foreground">Emp #:</span> {selected.employees?.employee_number}</div>
                <div><span className="text-muted-foreground">Department:</span> {selected.employees?.department}</div>
                <div><span className="text-muted-foreground">Designation:</span> {selected.employees?.designation}</div>
                <div><span className="text-muted-foreground">KRA PIN:</span> {selectedSensitive?.kra_pin ?? "—"}</div>
                <div><span className="text-muted-foreground">SHA #:</span> {selectedSensitive?.nhif_number ?? "—"}</div>
              </div>
              <Separator />
              <div>
                <h3 className="font-semibold mb-2">Earnings</h3>
                <div className="space-y-1">
                  <div className="flex justify-between"><span>Basic Salary</span><span className="font-mono">{fmt(selected.basic_salary)}</span></div>
                  {Number(selected.housing_allowance) > 0 && <div className="flex justify-between"><span>Housing Allowance</span><span className="font-mono">{fmt(selected.housing_allowance)}</span></div>}
                  {Number(selected.transport_allowance) > 0 && <div className="flex justify-between"><span>Transport Allowance</span><span className="font-mono">{fmt(selected.transport_allowance)}</span></div>}
                  {Number(selected.other_allowances) > 0 && <div className="flex justify-between"><span>Other Allowances</span><span className="font-mono">{fmt(selected.other_allowances)}</span></div>}
                  {Number(selected.overtime_pay) > 0 && <div className="flex justify-between"><span>Overtime</span><span className="font-mono">{fmt(selected.overtime_pay)}</span></div>}
                  {Number(selected.bonus) > 0 && <div className="flex justify-between"><span>Bonus</span><span className="font-mono">{fmt(selected.bonus)}</span></div>}
                  <div className="flex justify-between font-semibold border-t pt-1"><span>Gross Pay</span><span className="font-mono">{fmt(selected.gross_pay)}</span></div>
                </div>
              </div>
              <Separator />
              <div>
                <h3 className="font-semibold mb-2">Deductions</h3>
                <div className="space-y-1">
                  <div className="flex justify-between"><span>PAYE</span><span className="font-mono">{fmt(selected.paye)}</span></div>
                  <div className="flex justify-between"><span>SHA (2.75%)</span><span className="font-mono">{fmt(selected.nhif)}</span></div>
                  <div className="flex justify-between"><span>NSSF (Employee)</span><span className="font-mono">{fmt(selected.nssf_employee)}</span></div>
                  {Number(selected.housing_levy_employee ?? 0) > 0 && <div className="flex justify-between"><span>Housing Levy (1.5%)</span><span className="font-mono">{fmt(selected.housing_levy_employee)}</span></div>}
                  {Number(selected.loan_deduction) > 0 && <div className="flex justify-between"><span>Loan Deduction</span><span className="font-mono">{fmt(selected.loan_deduction)}</span></div>}
                  {Number(selected.sacco_deduction) > 0 && <div className="flex justify-between"><span>SACCO Deduction</span><span className="font-mono">{fmt(selected.sacco_deduction)}</span></div>}
                  {Number(selected.other_deductions) > 0 && <div className="flex justify-between"><span>Other Deductions</span><span className="font-mono">{fmt(selected.other_deductions)}</span></div>}
                  <div className="flex justify-between font-semibold border-t pt-1"><span>Total Deductions</span><span className="font-mono">{fmt(selected.total_deductions)}</span></div>
                </div>
              </div>
              <Separator />
              <div className="flex justify-between items-center text-lg font-bold bg-muted p-3 rounded-md">
                <span>Net Pay</span>
                <span className="font-mono">KES {fmt(selected.net_pay)}</span>
              </div>
              <div className="text-xs text-muted-foreground text-center pt-2">
                NSSF Employer: KES {fmt(selected.nssf_employer)} | Housing Levy Employer: KES {fmt(selected.housing_levy_employer ?? 0)} |
                Bank: {selectedSensitive?.bank_name ?? "—"} — {selectedSensitive?.bank_account ?? "—"}
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
}
