import { useEffect, useState } from "react";
import { supabase } from "@/integrations/supabase/client";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Button } from "@/components/ui/button";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Loader2, BarChart3, FileDown, Search } from "lucide-react";
import { Input } from "@/components/ui/input";
import { BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, PieChart, Pie, Cell, Legend } from "recharts";
import { generateP9PDF, buildP9MonthData } from "@/lib/p9-pdf";
import { useToast } from "@/hooks/use-toast";

const COLORS = [
  "hsl(210, 60%, 25%)",
  "hsl(175, 45%, 40%)",
  "hsl(38, 92%, 50%)",
  "hsl(142, 60%, 40%)",
  "hsl(0, 72%, 51%)",
  "hsl(260, 50%, 50%)",
];

interface PayrollRow {
  pay_period: string;
  gross_pay: number;
  total_deductions: number;
  net_pay: number;
  paye: number;
  nhif: number;
  nssf_employee: number;
  employees: { department: string; full_name: string } | null;
}

interface EmployeeBasic {
  id: string;
  employee_number: string;
  full_name: string;
  department: string;
}

export default function ReportsPage() {
  const [data, setData] = useState<PayrollRow[]>([]);
  const [loading, setLoading] = useState(true);
  const [period, setPeriod] = useState("");

  // P9 state
  const [employees, setEmployees] = useState<EmployeeBasic[]>([]);
  const [p9Year, setP9Year] = useState(new Date().getFullYear().toString());
  const [p9Search, setP9Search] = useState("");
  const [generatingP9, setGeneratingP9] = useState<string | null>(null);
  const { toast } = useToast();

  useEffect(() => {
    const fetchAll = async () => {
      setLoading(true);
      const [payrollRes, empRes] = await Promise.all([
        supabase
          .from("payroll")
          .select("pay_period, gross_pay, total_deductions, net_pay, paye, nhif, nssf_employee, employees(department, full_name)")
          .eq("status", "processed")
          .order("pay_period", { ascending: false }),
        supabase
          .from("employees")
          .select("id, employee_number, full_name, department")
          .eq("is_active", true)
          .order("full_name"),
      ]);
      if (payrollRes.data) setData(payrollRes.data as any);
      if (empRes.data) setEmployees(empRes.data as EmployeeBasic[]);
      setLoading(false);
    };
    fetchAll();
  }, []);

  const periods = [...new Set(data.map((d) => d.pay_period))].sort().reverse();
  const activePeriod = period || periods[0] || "";
  const periodData = data.filter((d) => d.pay_period === activePeriod);

  // Department summary
  const deptMap = new Map<string, { gross: number; deductions: number; net: number; count: number }>();
  periodData.forEach((r) => {
    const dept = r.employees?.department ?? "Unknown";
    const prev = deptMap.get(dept) ?? { gross: 0, deductions: 0, net: 0, count: 0 };
    deptMap.set(dept, {
      gross: prev.gross + Number(r.gross_pay),
      deductions: prev.deductions + Number(r.total_deductions),
      net: prev.net + Number(r.net_pay),
      count: prev.count + 1,
    });
  });
  const deptData = Array.from(deptMap.entries()).map(([dept, v]) => ({ department: dept, ...v }));

  // Deduction breakdown
  const totalPaye = periodData.reduce((s, r) => s + Number(r.paye), 0);
  const totalNhif = periodData.reduce((s, r) => s + Number(r.nhif), 0);
  const totalNssf = periodData.reduce((s, r) => s + Number(r.nssf_employee), 0);
  const deductionPie = [
    { name: "PAYE", value: totalPaye },
    { name: "NHIF", value: totalNhif },
    { name: "NSSF", value: totalNssf },
  ].filter((d) => d.value > 0);

  // Monthly trend (last 6 months)
  const trendData = periods.slice(0, 6).reverse().map((p) => {
    const rows = data.filter((d) => d.pay_period === p);
    return {
      period: p,
      gross: rows.reduce((s, r) => s + Number(r.gross_pay), 0),
      net: rows.reduce((s, r) => s + Number(r.net_pay), 0),
    };
  });

  const totalGross = periodData.reduce((s, r) => s + Number(r.gross_pay), 0);
  const totalNet = periodData.reduce((s, r) => s + Number(r.net_pay), 0);
  const totalDed = periodData.reduce((s, r) => s + Number(r.total_deductions), 0);
  const fmt = (n: number) => `KES ${n.toLocaleString("en-KE")}`;

  // P9 generation
  const handleGenerateP9 = async (emp: EmployeeBasic) => {
    setGeneratingP9(emp.id);
    try {
      // Fetch all payroll for this employee in selected year
      const { data: payrollRows, error } = await supabase
        .from("payroll")
        .select("*")
        .eq("employee_id", emp.id)
        .eq("status", "processed")
        .like("pay_period", `${p9Year}-%`)
        .order("pay_period");

      if (error) throw error;
      if (!payrollRows || payrollRows.length === 0) {
        toast({ title: "No data", description: `No processed payroll found for ${emp.full_name} in ${p9Year}.`, variant: "destructive" });
        setGeneratingP9(null);
        return;
      }

      // Fetch employee sensitive info for KRA PIN
      const { data: sensData } = await supabase.rpc("get_employee_sensitive_info", { p_employee_id: emp.id });
      const kraPin = sensData?.[0]?.kra_pin ?? "N/A";

      const months = buildP9MonthData(payrollRows);

      generateP9PDF({
        year: p9Year,
        employerName: "GIDEON WHEELS LTD",
        employerPin: "P0XXXXXXXXX", // Company KRA PIN - update as needed
        employeeName: emp.full_name,
        employeePin: kraPin,
        employeeNumber: emp.employee_number,
        months,
      });

      toast({ title: "P9 Generated", description: `P9 certificate for ${emp.full_name} (${p9Year}) downloaded.` });
    } catch (err: any) {
      toast({ title: "Error", description: err.message || "Failed to generate P9", variant: "destructive" });
    }
    setGeneratingP9(null);
  };

  // Available years from payroll data
  const availableYears = [...new Set(periods.map((p) => p.split("-")[0]))].sort().reverse();
  if (availableYears.length === 0) availableYears.push(new Date().getFullYear().toString());

  const filteredEmployees = employees.filter((e) =>
    e.full_name.toLowerCase().includes(p9Search.toLowerCase()) ||
    e.employee_number.toLowerCase().includes(p9Search.toLowerCase())
  );

  if (loading) return <div className="flex justify-center py-20"><Loader2 className="h-6 w-6 animate-spin text-primary" /></div>;

  return (
    <div>
      <div className="mb-6">
        <h1 className="text-2xl font-bold">Reports & Tax Certificates</h1>
        <p className="text-muted-foreground text-sm">Payroll reports, departmental summaries, and P9 tax certificates</p>
      </div>

      <Tabs defaultValue="reports" className="space-y-6">
        <TabsList>
          <TabsTrigger value="reports">
            <BarChart3 className="h-4 w-4 mr-2" /> Payroll Reports
          </TabsTrigger>
          <TabsTrigger value="p9">
            <FileDown className="h-4 w-4 mr-2" /> P9 Tax Certificates
          </TabsTrigger>
        </TabsList>

        {/* Payroll Reports Tab */}
        <TabsContent value="reports" className="space-y-6">
          <div className="flex justify-end">
            <Select value={activePeriod} onValueChange={setPeriod}>
              <SelectTrigger className="w-44"><SelectValue placeholder="Select period" /></SelectTrigger>
              <SelectContent>
                {periods.map((p) => <SelectItem key={p} value={p}>{p}</SelectItem>)}
              </SelectContent>
            </Select>
          </div>

          {/* Summary cards */}
          <div className="grid gap-4 sm:grid-cols-3">
            {[
              { label: "Total Gross Pay", value: fmt(totalGross) },
              { label: "Total Deductions", value: fmt(totalDed) },
              { label: "Total Net Pay", value: fmt(totalNet) },
            ].map((s) => (
              <Card key={s.label} className="stat-card">
                <CardContent className="p-5">
                  <p className="text-sm text-muted-foreground">{s.label}</p>
                  <p className="text-xl font-bold mt-1 font-mono">{s.value}</p>
                </CardContent>
              </Card>
            ))}
          </div>

          {/* Charts row */}
          <div className="grid gap-6 lg:grid-cols-2">
            <Card>
              <CardHeader><CardTitle className="text-sm">Payroll by Department</CardTitle></CardHeader>
              <CardContent>
                {deptData.length === 0 ? (
                  <p className="text-muted-foreground text-sm text-center py-8">No data for this period</p>
                ) : (
                  <ResponsiveContainer width="100%" height={280}>
                    <BarChart data={deptData}>
                      <CartesianGrid strokeDasharray="3 3" className="stroke-border" />
                      <XAxis dataKey="department" tick={{ fontSize: 11 }} />
                      <YAxis tick={{ fontSize: 11 }} />
                      <Tooltip formatter={(v: number) => `KES ${v.toLocaleString()}`} />
                      <Bar dataKey="gross" name="Gross Pay" fill="hsl(210, 60%, 25%)" radius={[4, 4, 0, 0]} />
                      <Bar dataKey="net" name="Net Pay" fill="hsl(175, 45%, 40%)" radius={[4, 4, 0, 0]} />
                    </BarChart>
                  </ResponsiveContainer>
                )}
              </CardContent>
            </Card>

            <Card>
              <CardHeader><CardTitle className="text-sm">Statutory Deductions Breakdown</CardTitle></CardHeader>
              <CardContent>
                {deductionPie.length === 0 ? (
                  <p className="text-muted-foreground text-sm text-center py-8">No data</p>
                ) : (
                  <ResponsiveContainer width="100%" height={280}>
                    <PieChart>
                      <Pie data={deductionPie} dataKey="value" nameKey="name" cx="50%" cy="50%" outerRadius={100} label={({ name, value }) => `${name}: KES ${value.toLocaleString()}`}>
                        {deductionPie.map((_, i) => <Cell key={i} fill={COLORS[i % COLORS.length]} />)}
                      </Pie>
                      <Legend />
                      <Tooltip formatter={(v: number) => `KES ${v.toLocaleString()}`} />
                    </PieChart>
                  </ResponsiveContainer>
                )}
              </CardContent>
            </Card>
          </div>

          {/* Monthly trend */}
          {trendData.length > 1 && (
            <Card>
              <CardHeader><CardTitle className="text-sm">Monthly Payroll Trend</CardTitle></CardHeader>
              <CardContent>
                <ResponsiveContainer width="100%" height={250}>
                  <BarChart data={trendData}>
                    <CartesianGrid strokeDasharray="3 3" className="stroke-border" />
                    <XAxis dataKey="period" tick={{ fontSize: 11 }} />
                    <YAxis tick={{ fontSize: 11 }} />
                    <Tooltip formatter={(v: number) => `KES ${v.toLocaleString()}`} />
                    <Bar dataKey="gross" name="Gross" fill="hsl(210, 60%, 25%)" radius={[4, 4, 0, 0]} />
                    <Bar dataKey="net" name="Net" fill="hsl(142, 60%, 40%)" radius={[4, 4, 0, 0]} />
                  </BarChart>
                </ResponsiveContainer>
              </CardContent>
            </Card>
          )}

          {/* Department table */}
          <Card>
            <CardHeader><CardTitle className="text-sm">Departmental Summary — {activePeriod}</CardTitle></CardHeader>
            <CardContent className="p-0">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Department</TableHead>
                    <TableHead className="text-right">Employees</TableHead>
                    <TableHead className="text-right">Gross Pay</TableHead>
                    <TableHead className="text-right">Deductions</TableHead>
                    <TableHead className="text-right">Net Pay</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {deptData.length === 0 ? (
                    <TableRow><TableCell colSpan={5} className="text-center py-6 text-muted-foreground">No records</TableCell></TableRow>
                  ) : (
                    <>
                      {deptData.map((d) => (
                        <TableRow key={d.department}>
                          <TableCell className="font-medium">{d.department}</TableCell>
                          <TableCell className="text-right">{d.count}</TableCell>
                          <TableCell className="text-right font-mono">{d.gross.toLocaleString()}</TableCell>
                          <TableCell className="text-right font-mono">{d.deductions.toLocaleString()}</TableCell>
                          <TableCell className="text-right font-mono font-semibold">{d.net.toLocaleString()}</TableCell>
                        </TableRow>
                      ))}
                      <TableRow className="bg-muted/50 font-semibold">
                        <TableCell>Total</TableCell>
                        <TableCell className="text-right">{periodData.length}</TableCell>
                        <TableCell className="text-right font-mono">{totalGross.toLocaleString()}</TableCell>
                        <TableCell className="text-right font-mono">{totalDed.toLocaleString()}</TableCell>
                        <TableCell className="text-right font-mono">{totalNet.toLocaleString()}</TableCell>
                      </TableRow>
                    </>
                  )}
                </TableBody>
              </Table>
            </CardContent>
          </Card>
        </TabsContent>

        {/* P9 Tax Certificates Tab */}
        <TabsContent value="p9" className="space-y-6">
          <Card>
            <CardHeader>
              <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
                <div>
                  <CardTitle className="text-lg">P9 Annual Tax Certificates</CardTitle>
                  <p className="text-sm text-muted-foreground mt-1">
                    Generate KRA P9 Tax Deduction Cards for employees — includes monthly PAYE breakdown, personal relief, and insurance relief.
                  </p>
                </div>
                <Select value={p9Year} onValueChange={setP9Year}>
                  <SelectTrigger className="w-32"><SelectValue placeholder="Year" /></SelectTrigger>
                  <SelectContent>
                    {availableYears.map((y) => <SelectItem key={y} value={y}>{y}</SelectItem>)}
                  </SelectContent>
                </Select>
              </div>
            </CardHeader>
            <CardContent>
              <div className="relative mb-4 max-w-sm">
                <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input placeholder="Search employees..." className="pl-9" value={p9Search} onChange={(e) => setP9Search(e.target.value)} />
              </div>

              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Employee #</TableHead>
                    <TableHead>Name</TableHead>
                    <TableHead>Department</TableHead>
                    <TableHead className="text-right">Action</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredEmployees.length === 0 ? (
                    <TableRow><TableCell colSpan={4} className="text-center py-8 text-muted-foreground">No employees found</TableCell></TableRow>
                  ) : (
                    filteredEmployees.map((emp) => (
                      <TableRow key={emp.id}>
                        <TableCell className="font-mono text-sm">{emp.employee_number}</TableCell>
                        <TableCell className="font-medium">{emp.full_name}</TableCell>
                        <TableCell>{emp.department}</TableCell>
                        <TableCell className="text-right">
                          <Button
                            size="sm"
                            variant="outline"
                            onClick={() => handleGenerateP9(emp)}
                            disabled={generatingP9 === emp.id}
                          >
                            {generatingP9 === emp.id ? (
                              <Loader2 className="h-3.5 w-3.5 mr-1 animate-spin" />
                            ) : (
                              <FileDown className="h-3.5 w-3.5 mr-1" />
                            )}
                            Download P9
                          </Button>
                        </TableCell>
                      </TableRow>
                    ))
                  )}
                </TableBody>
              </Table>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}
