import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers":
    "authorization, x-client-info, apikey, content-type, x-supabase-client-platform, x-supabase-client-platform-version, x-supabase-client-runtime, x-supabase-client-runtime-version",
};

const fmt = (n: number) =>
  `KES ${Number(n).toLocaleString("en-KE", { minimumFractionDigits: 2 })}`;

const UUID_REGEX = /^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i;
const EMAIL_REGEX = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;

serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const RESEND_API_KEY = Deno.env.get("RESEND_API_KEY");
    if (!RESEND_API_KEY) {
      throw new Error("RESEND_API_KEY is not configured");
    }

    const supabaseUrl = Deno.env.get("SUPABASE_URL")!;
    const supabaseServiceKey = Deno.env.get("SUPABASE_SERVICE_ROLE_KEY")!;
    const supabase = createClient(supabaseUrl, supabaseServiceKey);

    // Verify the caller is authenticated
    const authHeader = req.headers.get("Authorization");
    if (!authHeader) throw new Error("Not authenticated");

    const token = authHeader.replace("Bearer ", "");
    const { data: { user }, error: authError } = await supabase.auth.getUser(token);
    if (authError || !user) throw new Error("Authentication failed");

    // Verify role-based authorization
    const { data: userRoles } = await supabase
      .from("user_roles")
      .select("role")
      .eq("user_id", user.id);

    const hasPermission = userRoles?.some((r: { role: string }) =>
      ["admin", "hr_officer", "accountant"].includes(r.role)
    );
    if (!hasPermission) {
      throw new Error("Insufficient permissions: requires admin, HR, or accountant role");
    }

    // Validate inputs
    const body = await req.json();
    const { payrollId, recipientEmail } = body;

    if (!payrollId || typeof payrollId !== "string" || !UUID_REGEX.test(payrollId)) {
      throw new Error("Invalid payrollId: must be a valid UUID");
    }

    if (recipientEmail && (typeof recipientEmail !== "string" || !EMAIL_REGEX.test(recipientEmail) || recipientEmail.length > 255)) {
      throw new Error("Invalid recipientEmail: must be a valid email address");
    }

    // Fetch payroll with non-sensitive employee data
    const { data: payroll, error: fetchErr } = await supabase
      .from("payroll")
      .select("*, employees(employee_number, full_name, department, designation, email)")
      .eq("id", payrollId)
      .single();

    if (fetchErr || !payroll) throw new Error("Payroll record not found");

    const emp = payroll.employees as any;
    const toEmail = recipientEmail || emp?.email;
    if (!toEmail) throw new Error("No email address found for this employee");

    if (!EMAIL_REGEX.test(toEmail)) {
      throw new Error("Invalid email address on employee record");
    }

    // Fetch sensitive data from separate table (service role has access)
    const { data: sensData } = await supabase
      .from("employee_sensitive_data")
      .select("kra_pin, nhif_number, nssf_number, bank_name, bank_account")
      .eq("employee_id", payroll.employee_id)
      .single();

    const kraPin = sensData?.kra_pin ?? "—";
    const nhifNumber = sensData?.nhif_number ?? "—";
    const bankInfo = sensData ? `${sensData.bank_name ?? "—"} — ${sensData.bank_account ?? "—"}` : "—";

    const html = `
      <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; color: #1a1a2e;">
        <div style="background: #143c5a; color: white; padding: 20px; text-align: center;">
          <h1 style="margin: 0; font-size: 20px;">GIDEON WHEELS LTD</h1>
          <p style="margin: 4px 0 0; opacity: 0.8; font-size: 14px;">Payslip for ${payroll.pay_period}</p>
        </div>
        <div style="padding: 20px; border: 1px solid #e0e0e0;">
          <table style="width: 100%; font-size: 13px; margin-bottom: 16px;">
            <tr><td style="color: #666;">Name:</td><td><strong>${emp?.full_name}</strong></td><td style="color: #666;">Emp #:</td><td>${emp?.employee_number}</td></tr>
            <tr><td style="color: #666;">Department:</td><td>${emp?.department}</td><td style="color: #666;">Designation:</td><td>${emp?.designation}</td></tr>
            <tr><td style="color: #666;">KRA PIN:</td><td>${kraPin}</td><td style="color: #666;">NHIF #:</td><td>${nhifNumber}</td></tr>
          </table>
          <h3 style="border-bottom: 2px solid #143c5a; padding-bottom: 4px; color: #143c5a;">Earnings</h3>
          <table style="width: 100%; font-size: 13px;">
            <tr><td>Basic Salary</td><td style="text-align: right;">${fmt(payroll.basic_salary)}</td></tr>
            ${payroll.housing_allowance > 0 ? `<tr><td>Housing Allowance</td><td style="text-align: right;">${fmt(payroll.housing_allowance)}</td></tr>` : ""}
            ${payroll.transport_allowance > 0 ? `<tr><td>Transport Allowance</td><td style="text-align: right;">${fmt(payroll.transport_allowance)}</td></tr>` : ""}
            ${payroll.other_allowances > 0 ? `<tr><td>Other Allowances</td><td style="text-align: right;">${fmt(payroll.other_allowances)}</td></tr>` : ""}
            ${payroll.overtime_pay > 0 ? `<tr><td>Overtime</td><td style="text-align: right;">${fmt(payroll.overtime_pay)}</td></tr>` : ""}
            ${payroll.bonus > 0 ? `<tr><td>Bonus</td><td style="text-align: right;">${fmt(payroll.bonus)}</td></tr>` : ""}
            <tr style="font-weight: bold; border-top: 1px solid #ccc;"><td>Gross Pay</td><td style="text-align: right;">${fmt(payroll.gross_pay)}</td></tr>
          </table>
          <h3 style="border-bottom: 2px solid #8c2828; padding-bottom: 4px; color: #8c2828; margin-top: 16px;">Deductions</h3>
          <table style="width: 100%; font-size: 13px;">
            <tr><td>PAYE</td><td style="text-align: right;">${fmt(payroll.paye)}</td></tr>
            <tr><td>NHIF</td><td style="text-align: right;">${fmt(payroll.nhif)}</td></tr>
            <tr><td>NSSF (Employee)</td><td style="text-align: right;">${fmt(payroll.nssf_employee)}</td></tr>
            ${payroll.loan_deduction > 0 ? `<tr><td>Loan</td><td style="text-align: right;">${fmt(payroll.loan_deduction)}</td></tr>` : ""}
            ${payroll.sacco_deduction > 0 ? `<tr><td>SACCO</td><td style="text-align: right;">${fmt(payroll.sacco_deduction)}</td></tr>` : ""}
            <tr style="font-weight: bold; border-top: 1px solid #ccc;"><td>Total Deductions</td><td style="text-align: right;">${fmt(payroll.total_deductions)}</td></tr>
          </table>
          <div style="background: #e6f5fa; padding: 12px; margin-top: 16px; text-align: center; border-radius: 6px;">
            <span style="font-size: 18px; font-weight: bold; color: #143c5a;">Net Pay: ${fmt(payroll.net_pay)}</span>
          </div>
          <p style="font-size: 11px; color: #999; margin-top: 12px; text-align: center;">
            NSSF Employer Contribution: ${fmt(payroll.nssf_employer)} | Bank: ${bankInfo} | This is a computer-generated payslip.
          </p>
        </div>
      </div>
    `;

    const emailRes = await fetch("https://api.resend.com/emails", {
      method: "POST",
      headers: {
        Authorization: `Bearer ${RESEND_API_KEY}`,
        "Content-Type": "application/json",
      },
      body: JSON.stringify({
        from: "Gideon Wheels Payroll <onboarding@resend.dev>",
        to: [toEmail],
        subject: `Payslip — ${payroll.pay_period} | ${emp?.full_name}`,
        html,
      }),
    });

    const emailData = await emailRes.json();
    if (!emailRes.ok) {
      const detail = emailData?.message || emailData?.name || JSON.stringify(emailData);
      if (emailRes.status === 403 || detail.toLowerCase().includes("verify")) {
        throw new Error(`Resend requires a verified domain to send to this address. On the free plan you can only send to your own Resend account email. Details: ${detail}`);
      }
      throw new Error(`Email service error (${emailRes.status}): ${detail}`);
    }

    // Log the email send in audit
    await supabase.from("audit_logs").insert({
      action: "payslip_emailed",
      table_name: "payroll",
      record_id: payrollId,
      user_id: user.id,
      new_data: { recipient: toEmail, resend_id: emailData.id },
    });

    return new Response(
      JSON.stringify({ success: true, emailId: emailData.id, sentTo: toEmail }),
      { status: 200, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  } catch (error: unknown) {
    console.error("Error sending payslip email:", error);
    const message = error instanceof Error ? error.message : "An error occurred";
    const status = message.includes("Insufficient permissions") ? 403 :
                   message.includes("Not authenticated") || message.includes("Authentication failed") ? 401 :
                   message.includes("Invalid") ? 400 : 500;
    return new Response(
      JSON.stringify({ success: false, error: message }),
      { status, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  }
});
