
-- =============================================
-- PAYROLL MANAGEMENT SYSTEM - DATABASE SCHEMA
-- Gideon Wheels Ltd
-- =============================================

-- 1. Create role enum
CREATE TYPE public.app_role AS ENUM ('admin', 'hr_officer', 'accountant');

-- 2. Profiles table
CREATE TABLE public.profiles (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID REFERENCES auth.users(id) ON DELETE CASCADE NOT NULL UNIQUE,
  full_name TEXT NOT NULL DEFAULT '',
  email TEXT NOT NULL DEFAULT '',
  created_at TIMESTAMPTZ NOT NULL DEFAULT now(),
  updated_at TIMESTAMPTZ NOT NULL DEFAULT now()
);

-- 3. User roles table (separate from profiles for security)
CREATE TABLE public.user_roles (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID REFERENCES auth.users(id) ON DELETE CASCADE NOT NULL,
  role app_role NOT NULL,
  created_at TIMESTAMPTZ NOT NULL DEFAULT now(),
  UNIQUE(user_id, role)
);

-- 4. Employees table
CREATE TABLE public.employees (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  employee_number TEXT NOT NULL UNIQUE,
  full_name TEXT NOT NULL,
  email TEXT,
  phone TEXT,
  department TEXT NOT NULL DEFAULT 'General',
  designation TEXT NOT NULL DEFAULT '',
  basic_salary NUMERIC(12,2) NOT NULL DEFAULT 0,
  kra_pin TEXT,
  nhif_number TEXT,
  nssf_number TEXT,
  bank_name TEXT,
  bank_account TEXT,
  bank_branch TEXT,
  date_joined DATE NOT NULL DEFAULT CURRENT_DATE,
  is_active BOOLEAN NOT NULL DEFAULT true,
  created_at TIMESTAMPTZ NOT NULL DEFAULT now(),
  updated_at TIMESTAMPTZ NOT NULL DEFAULT now()
);

-- 5. Tax rates table (Kenyan PAYE brackets)
CREATE TABLE public.tax_rates (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  bracket_name TEXT NOT NULL,
  min_amount NUMERIC(12,2) NOT NULL DEFAULT 0,
  max_amount NUMERIC(12,2),
  rate_percentage NUMERIC(5,2) NOT NULL,
  effective_date DATE NOT NULL DEFAULT CURRENT_DATE,
  is_active BOOLEAN NOT NULL DEFAULT true,
  created_at TIMESTAMPTZ NOT NULL DEFAULT now(),
  updated_at TIMESTAMPTZ NOT NULL DEFAULT now()
);

-- 6. NHIF rates table
CREATE TABLE public.nhif_rates (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  min_salary NUMERIC(12,2) NOT NULL DEFAULT 0,
  max_salary NUMERIC(12,2),
  monthly_contribution NUMERIC(12,2) NOT NULL,
  is_active BOOLEAN NOT NULL DEFAULT true,
  created_at TIMESTAMPTZ NOT NULL DEFAULT now()
);

-- 7. Payroll table
CREATE TABLE public.payroll (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  employee_id UUID REFERENCES public.employees(id) ON DELETE CASCADE NOT NULL,
  pay_period TEXT NOT NULL, -- e.g., '2026-02'
  basic_salary NUMERIC(12,2) NOT NULL DEFAULT 0,
  housing_allowance NUMERIC(12,2) NOT NULL DEFAULT 0,
  transport_allowance NUMERIC(12,2) NOT NULL DEFAULT 0,
  other_allowances NUMERIC(12,2) NOT NULL DEFAULT 0,
  overtime_pay NUMERIC(12,2) NOT NULL DEFAULT 0,
  bonus NUMERIC(12,2) NOT NULL DEFAULT 0,
  gross_pay NUMERIC(12,2) NOT NULL DEFAULT 0,
  paye NUMERIC(12,2) NOT NULL DEFAULT 0,
  nhif NUMERIC(12,2) NOT NULL DEFAULT 0,
  nssf_employee NUMERIC(12,2) NOT NULL DEFAULT 0,
  nssf_employer NUMERIC(12,2) NOT NULL DEFAULT 0,
  loan_deduction NUMERIC(12,2) NOT NULL DEFAULT 0,
  sacco_deduction NUMERIC(12,2) NOT NULL DEFAULT 0,
  other_deductions NUMERIC(12,2) NOT NULL DEFAULT 0,
  total_deductions NUMERIC(12,2) NOT NULL DEFAULT 0,
  net_pay NUMERIC(12,2) NOT NULL DEFAULT 0,
  processed_by UUID REFERENCES auth.users(id),
  processed_at TIMESTAMPTZ NOT NULL DEFAULT now(),
  status TEXT NOT NULL DEFAULT 'draft',
  created_at TIMESTAMPTZ NOT NULL DEFAULT now(),
  updated_at TIMESTAMPTZ NOT NULL DEFAULT now(),
  UNIQUE(employee_id, pay_period)
);

-- 8. Audit logs table
CREATE TABLE public.audit_logs (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID REFERENCES auth.users(id),
  action TEXT NOT NULL,
  table_name TEXT NOT NULL,
  record_id TEXT,
  old_data JSONB,
  new_data JSONB,
  ip_address TEXT,
  created_at TIMESTAMPTZ NOT NULL DEFAULT now()
);

-- =============================================
-- ENABLE RLS ON ALL TABLES
-- =============================================
ALTER TABLE public.profiles ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.user_roles ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.employees ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.tax_rates ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.nhif_rates ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.payroll ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.audit_logs ENABLE ROW LEVEL SECURITY;

-- =============================================
-- SECURITY DEFINER HELPER FUNCTIONS
-- =============================================

-- Check if user has a specific role
CREATE OR REPLACE FUNCTION public.has_role(_user_id UUID, _role app_role)
RETURNS BOOLEAN
LANGUAGE sql
STABLE
SECURITY DEFINER
SET search_path = public
AS $$
  SELECT EXISTS (
    SELECT 1 FROM public.user_roles
    WHERE user_id = _user_id AND role = _role
  )
$$;

-- Check if user is admin
CREATE OR REPLACE FUNCTION public.is_admin()
RETURNS BOOLEAN
LANGUAGE sql
STABLE
SECURITY DEFINER
SET search_path = public
AS $$
  SELECT public.has_role(auth.uid(), 'admin')
$$;

-- Check if user is HR officer
CREATE OR REPLACE FUNCTION public.is_hr()
RETURNS BOOLEAN
LANGUAGE sql
STABLE
SECURITY DEFINER
SET search_path = public
AS $$
  SELECT public.has_role(auth.uid(), 'hr_officer')
$$;

-- Check if user is accountant
CREATE OR REPLACE FUNCTION public.is_accountant()
RETURNS BOOLEAN
LANGUAGE sql
STABLE
SECURITY DEFINER
SET search_path = public
AS $$
  SELECT public.has_role(auth.uid(), 'accountant')
$$;

-- =============================================
-- RLS POLICIES
-- =============================================

-- PROFILES
CREATE POLICY "Users can view own profile" ON public.profiles FOR SELECT USING (auth.uid() = user_id);
CREATE POLICY "Admins can view all profiles" ON public.profiles FOR SELECT USING (public.is_admin());
CREATE POLICY "Users can update own profile" ON public.profiles FOR UPDATE USING (auth.uid() = user_id);
CREATE POLICY "Admins can manage profiles" ON public.profiles FOR ALL USING (public.is_admin());
CREATE POLICY "System can create profiles" ON public.profiles FOR INSERT WITH CHECK (auth.uid() = user_id);

-- USER ROLES (only admins manage roles)
CREATE POLICY "Admins can manage roles" ON public.user_roles FOR ALL USING (public.is_admin());
CREATE POLICY "Users can view own roles" ON public.user_roles FOR SELECT USING (auth.uid() = user_id);

-- EMPLOYEES
CREATE POLICY "Admins full access employees" ON public.employees FOR ALL USING (public.is_admin());
CREATE POLICY "HR full access employees" ON public.employees FOR ALL USING (public.is_hr());
CREATE POLICY "Accountants can view employees" ON public.employees FOR SELECT USING (public.is_accountant());

-- TAX RATES
CREATE POLICY "All authenticated can view tax rates" ON public.tax_rates FOR SELECT TO authenticated USING (true);
CREATE POLICY "Admins can manage tax rates" ON public.tax_rates FOR ALL USING (public.is_admin());

-- NHIF RATES
CREATE POLICY "All authenticated can view nhif rates" ON public.nhif_rates FOR SELECT TO authenticated USING (true);
CREATE POLICY "Admins can manage nhif rates" ON public.nhif_rates FOR ALL USING (public.is_admin());

-- PAYROLL
CREATE POLICY "Admins full access payroll" ON public.payroll FOR ALL USING (public.is_admin());
CREATE POLICY "HR can view payroll" ON public.payroll FOR SELECT USING (public.is_hr());
CREATE POLICY "Accountants full access payroll" ON public.payroll FOR ALL USING (public.is_accountant());

-- AUDIT LOGS
CREATE POLICY "Admins can view audit logs" ON public.audit_logs FOR SELECT USING (public.is_admin());
CREATE POLICY "Authenticated can insert audit logs" ON public.audit_logs FOR INSERT TO authenticated WITH CHECK (auth.uid() = user_id);

-- =============================================
-- TRIGGERS
-- =============================================

-- Auto-create profile on signup
CREATE OR REPLACE FUNCTION public.handle_new_user()
RETURNS TRIGGER
LANGUAGE plpgsql
SECURITY DEFINER
SET search_path = public
AS $$
BEGIN
  INSERT INTO public.profiles (user_id, full_name, email)
  VALUES (NEW.id, COALESCE(NEW.raw_user_meta_data->>'full_name', ''), NEW.email);
  RETURN NEW;
END;
$$;

CREATE TRIGGER on_auth_user_created
  AFTER INSERT ON auth.users
  FOR EACH ROW EXECUTE FUNCTION public.handle_new_user();

-- Updated_at triggers
CREATE OR REPLACE FUNCTION public.update_updated_at()
RETURNS TRIGGER
LANGUAGE plpgsql
SET search_path = public
AS $$
BEGIN
  NEW.updated_at = now();
  RETURN NEW;
END;
$$;

CREATE TRIGGER update_profiles_updated_at BEFORE UPDATE ON public.profiles FOR EACH ROW EXECUTE FUNCTION public.update_updated_at();
CREATE TRIGGER update_employees_updated_at BEFORE UPDATE ON public.employees FOR EACH ROW EXECUTE FUNCTION public.update_updated_at();
CREATE TRIGGER update_payroll_updated_at BEFORE UPDATE ON public.payroll FOR EACH ROW EXECUTE FUNCTION public.update_updated_at();
CREATE TRIGGER update_tax_rates_updated_at BEFORE UPDATE ON public.tax_rates FOR EACH ROW EXECUTE FUNCTION public.update_updated_at();

-- =============================================
-- SEED KENYAN TAX RATES (2024/2025)
-- =============================================
INSERT INTO public.tax_rates (bracket_name, min_amount, max_amount, rate_percentage, effective_date) VALUES
  ('First Band', 0, 24000, 10.00, '2024-01-01'),
  ('Second Band', 24001, 32333, 25.00, '2024-01-01'),
  ('Third Band', 32334, 500000, 30.00, '2024-01-01'),
  ('Fourth Band', 500001, 800000, 32.50, '2024-01-01'),
  ('Excess', 800001, NULL, 35.00, '2024-01-01');

-- NHIF rates
INSERT INTO public.nhif_rates (min_salary, max_salary, monthly_contribution) VALUES
  (0, 5999, 150),
  (6000, 7999, 300),
  (8000, 11999, 400),
  (12000, 14999, 500),
  (15000, 19999, 600),
  (20000, 24999, 750),
  (25000, 29999, 850),
  (30000, 34999, 900),
  (35000, 39999, 950),
  (40000, 44999, 1000),
  (45000, 49999, 1100),
  (50000, 59999, 1200),
  (60000, 69999, 1300),
  (70000, 79999, 1400),
  (80000, 89999, 1500),
  (90000, 99999, 1600),
  (100000, NULL, 1700);
