
-- Create table for sensitive employee data (bank, tax IDs)
CREATE TABLE public.employee_sensitive_data (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  employee_id UUID NOT NULL UNIQUE REFERENCES public.employees(id) ON DELETE CASCADE,
  bank_account TEXT,
  bank_branch TEXT,
  bank_name TEXT,
  kra_pin TEXT,
  nhif_number TEXT,
  nssf_number TEXT,
  mpesa_number TEXT,
  created_at TIMESTAMPTZ NOT NULL DEFAULT now(),
  updated_at TIMESTAMPTZ NOT NULL DEFAULT now()
);

-- Migrate existing data
INSERT INTO public.employee_sensitive_data (employee_id, bank_account, bank_branch, bank_name, kra_pin, nhif_number, nssf_number, mpesa_number)
SELECT id, bank_account, bank_branch, bank_name, kra_pin, nhif_number, nssf_number, mpesa_number
FROM public.employees;

-- Drop sensitive columns from employees table
ALTER TABLE public.employees DROP COLUMN bank_account;
ALTER TABLE public.employees DROP COLUMN bank_branch;
ALTER TABLE public.employees DROP COLUMN bank_name;
ALTER TABLE public.employees DROP COLUMN kra_pin;
ALTER TABLE public.employees DROP COLUMN nhif_number;
ALTER TABLE public.employees DROP COLUMN nssf_number;
ALTER TABLE public.employees DROP COLUMN mpesa_number;

-- Enable RLS - admin and HR only
ALTER TABLE public.employee_sensitive_data ENABLE ROW LEVEL SECURITY;

CREATE POLICY "Admins full access sensitive data"
ON public.employee_sensitive_data FOR ALL
USING (public.is_admin());

CREATE POLICY "HR full access sensitive data"
ON public.employee_sensitive_data FOR ALL
USING (public.is_hr());

-- Trigger for updated_at
CREATE TRIGGER update_employee_sensitive_data_updated_at
BEFORE UPDATE ON public.employee_sensitive_data
FOR EACH ROW
EXECUTE FUNCTION public.update_updated_at();

-- RPC function for payslip generation - allows accountants to get sensitive data for a specific employee
-- This is more controlled than direct table access (single record, read-only, role-checked)
CREATE OR REPLACE FUNCTION public.get_employee_sensitive_info(p_employee_id UUID)
RETURNS TABLE(
  bank_account TEXT,
  bank_branch TEXT,
  bank_name TEXT,
  kra_pin TEXT,
  nhif_number TEXT,
  nssf_number TEXT,
  mpesa_number TEXT
)
LANGUAGE SQL
STABLE
SECURITY DEFINER
SET search_path = public
AS $$
  SELECT esd.bank_account, esd.bank_branch, esd.bank_name, esd.kra_pin, esd.nhif_number, esd.nssf_number, esd.mpesa_number
  FROM employee_sensitive_data esd
  WHERE esd.employee_id = p_employee_id
    AND (public.is_admin() OR public.is_hr() OR public.is_accountant())
$$;
